<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Prevent PHP notices/warnings from corrupting JSON output
error_reporting(E_ALL);
ini_set('display_errors', '0');
ini_set('log_errors', '1');

ob_start();

try {
    require_once __DIR__ . '/../config/database.php';
    require_once __DIR__ . '/../includes/ResumeParser.php';

    $action = $_POST['action'] ?? '';

    if ($action === 'login') {
        handleLogin($conn);
    } elseif ($action === 'signup') {
        handleSignup($conn);
    } else {
        echo json_encode(['success' => false, 'message' => 'Invalid action']);
    }
} catch (Throwable $e) {
    ob_end_clean();
    $message = 'An error occurred. Please try again.';
    if (!empty($_GET['debug']) || (isset($_POST['debug']) && $_POST['debug'])) {
        $message = $e->getMessage() . ' (in ' . $e->getFile() . ':' . $e->getLine() . ')';
    }
    echo json_encode(['success' => false, 'message' => $message]);
}

function handleLogin($conn) {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';

    if (empty($username) || empty($password)) {
        echo json_encode(['success' => false, 'message' => 'Username and password are required']);
        return;
    }

    $stmt = $conn->prepare('SELECT id, username, password FROM users WHERE username = ?');
    $stmt->bind_param('s', $username);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        echo json_encode(['success' => false, 'message' => 'Invalid username or password']);
        return;
    }

    $user = $result->fetch_assoc();
    if (!password_verify($password, $user['password'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid username or password']);
        return;
    }

    session_start();
    $_SESSION['user_id'] = $user['id'];
    $_SESSION['username'] = $user['username'];

    echo json_encode([
        'success' => true,
        'message' => 'Login successful',
        'redirect' => 'profile.php'
    ]);
}

function handleSignup($conn) {
    $username = trim($_POST['username'] ?? '');
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $phone = trim($_POST['phone'] ?? '');
    $signature_data = $_POST['signature_data'] ?? '';

    if (empty($username) || empty($first_name) || empty($last_name) || empty($email) || empty($password) || empty($phone) || empty($signature_data)) {
        echo json_encode(['success' => false, 'message' => 'All required fields must be filled']);
        return;
    }

    // Validate email
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['success' => false, 'message' => 'Invalid email address']);
        return;
    }

    // Check username exists
    $stmt = $conn->prepare('SELECT id FROM users WHERE username = ?');
    $stmt->bind_param('s', $username);
    $stmt->execute();
    if ($stmt->get_result()->num_rows > 0) {
        echo json_encode(['success' => false, 'message' => 'Username already exists']);
        return;
    }

    // Check email exists
    $stmt = $conn->prepare('SELECT id FROM users WHERE email = ?');
    $stmt->bind_param('s', $email);
    $stmt->execute();
    if ($stmt->get_result()->num_rows > 0) {
        echo json_encode(['success' => false, 'message' => 'Email already registered']);
        return;
    }

    // Handle resume upload
    $resume_path = null;
    if (isset($_FILES['resume']) && $_FILES['resume']['error'] === UPLOAD_ERR_OK) {
        $allowed = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
        $ext = strtolower(pathinfo($_FILES['resume']['name'], PATHINFO_EXTENSION));
        if (!in_array($_FILES['resume']['type'], $allowed) && !in_array($ext, ['pdf', 'doc', 'docx'])) {
            echo json_encode(['success' => false, 'message' => 'Resume must be PDF or Word file']);
            return;
        }
        $uploads_dir = __DIR__ . '/../uploads';
        if (!is_dir($uploads_dir)) {
            mkdir($uploads_dir, 0755, true);
        }
        // Rename file to: FirstName_LastName.pdf (or .doc/.docx)
        $safe_first = preg_replace('/[^a-zA-Z0-9\s\-]/', '', $first_name);
        $safe_first = preg_replace('/\s+/', '_', trim($safe_first)) ?: 'first';
        $safe_last = preg_replace('/[^a-zA-Z0-9\s\-]/', '', $last_name);
        $safe_last = preg_replace('/\s+/', '_', trim($safe_last)) ?: 'last';
        $resume_name = $safe_first . '_' . $safe_last . '.' . $ext;
        // If duplicate name exists, append uniqid to avoid overwriting
        $base_path = __DIR__ . '/../uploads/' . $resume_name;
        if (file_exists($base_path)) {
            $resume_name = $safe_first . '_' . $safe_last . '_' . uniqid() . '.' . $ext;
        }
        $resume_path = 'uploads/' . $resume_name;
        if (!move_uploaded_file($_FILES['resume']['tmp_name'], __DIR__ . '/../' . $resume_path)) {
            echo json_encode(['success' => false, 'message' => 'Failed to upload resume']);
            return;
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'Please upload your resume (PDF or Word)']);
        return;
    }

    // Save signature
    $signature_path = null;
    if (!empty($signature_data) && preg_match('/^data:image\/png;base64,/', $signature_data)) {
        $uploads_dir = __DIR__ . '/../uploads';
        if (!is_dir($uploads_dir)) mkdir($uploads_dir, 0755, true);
        $signature_name = 'signature_' . uniqid() . '.png';
        $signature_path = 'uploads/' . $signature_name;
        $data = base64_decode(preg_replace('/^data:image\/\w+;base64,/', '', $signature_data));
        if ($data && file_put_contents(__DIR__ . '/../' . $signature_path, $data) === false) {
            echo json_encode(['success' => false, 'message' => 'Failed to save signature']);
            return;
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'Please draw your signature']);
        return;
    }

    $password_hash = password_hash($password, PASSWORD_DEFAULT);

    $stmt = $conn->prepare('INSERT INTO users (username, password, first_name, last_name, email, phone, signature_path, resume_path) VALUES (?, ?, ?, ?, ?, ?, ?, ?)');
    $stmt->bind_param('ssssssss', $username, $password_hash, $first_name, $last_name, $email, $phone, $signature_path, $resume_path);

    if ($stmt->execute()) {
        $userId = $conn->insert_id;
        session_start();
        $_SESSION['user_id'] = $userId;
        $_SESSION['username'] = $username;
        
        // Parse PDF resume and store profile data in database
        if ($ext === 'pdf' && $resume_path) {
            $resumeFullPath = __DIR__ . '/../' . $resume_path;
            if (file_exists($resumeFullPath)) {
                try {
                    $parser = new ResumeParser($conn);
                    $parsedData = $parser->parsePdf($resumeFullPath, $first_name, $last_name);
                    if ($parsedData) {
                        $parser->storeInDatabase($userId, $parsedData, [
                            'first_name' => $first_name,
                            'last_name' => $last_name
                        ]);
                    }
                } catch (Exception $e) {
                    // Continue - profile data can be added manually later
                }
            }
        }
        
        echo json_encode([
            'success' => true,
            'message' => 'Signup successful! Redirecting...',
            'redirect' => 'profile.php'
        ]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Registration failed. Please try again.']);
    }
}
