// Auth page - Login / Signup tab switching and signature canvas
document.addEventListener('DOMContentLoaded', function() {
    const tabs = document.querySelectorAll('.auth-tab');
    const loginForm = document.getElementById('loginForm');
    const signupForm = document.getElementById('signupForm');

    // Tab switching
    tabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const target = this.dataset.tab;
            tabs.forEach(t => t.classList.remove('active'));
            this.classList.add('active');

            if (target === 'login') {
                loginForm.classList.add('active');
                signupForm.classList.remove('active');
                document.getElementById('loginMessage').classList.remove('error', 'success');
                document.getElementById('loginMessage').textContent = '';
            } else {
                signupForm.classList.add('active');
                loginForm.classList.remove('active');
                document.getElementById('signupMessage').classList.remove('error', 'success');
                document.getElementById('signupMessage').textContent = '';
            }
        });
    });

    // Signature canvas
    const canvas = document.getElementById('signatureCanvas');
    const ctx = canvas.getContext('2d');
    const clearBtn = document.getElementById('clearSignature');
    const signatureInput = document.getElementById('signatureData');

    let isDrawing = false;
    let lastX = 0;
    let lastY = 0;

    // Set canvas background and stroke style
    ctx.fillStyle = '#ffffff';
    ctx.fillRect(0, 0, canvas.width, canvas.height);
    ctx.strokeStyle = '#000000';
    ctx.lineWidth = 2;
    ctx.lineCap = 'round';

    function getPos(e) {
        const rect = canvas.getBoundingClientRect();
        const scaleX = canvas.width / rect.width;
        const scaleY = canvas.height / rect.height;
        const clientX = e.touches ? e.touches[0].clientX : e.clientX;
        const clientY = e.touches ? e.touches[0].clientY : e.clientY;
        return {
            x: (clientX - rect.left) * scaleX,
            y: (clientY - rect.top) * scaleY
        };
    }

    function startDrawing(e) {
        e.preventDefault();
        isDrawing = true;
        const pos = getPos(e);
        lastX = pos.x;
        lastY = pos.y;
    }

    function draw(e) {
        e.preventDefault();
        if (!isDrawing) return;
        const pos = getPos(e);
        ctx.beginPath();
        ctx.moveTo(lastX, lastY);
        ctx.lineTo(pos.x, pos.y);
        ctx.stroke();
        lastX = pos.x;
        lastY = pos.y;
        updateSignatureData();
    }

    function stopDrawing() {
        isDrawing = false;
    }

    function updateSignatureData() {
        signatureInput.value = canvas.toDataURL('image/png');
    }

    canvas.addEventListener('mousedown', startDrawing);
    canvas.addEventListener('mousemove', draw);
    canvas.addEventListener('mouseup', stopDrawing);
    canvas.addEventListener('mouseleave', stopDrawing);
    canvas.addEventListener('touchstart', startDrawing, { passive: false });
    canvas.addEventListener('touchmove', draw, { passive: false });
    canvas.addEventListener('touchend', stopDrawing);

    clearBtn.addEventListener('click', function() {
        ctx.fillStyle = '#ffffff';
        ctx.fillRect(0, 0, canvas.width, canvas.height);
        signatureInput.value = '';
    });


    // AJAX form submission
    loginForm.addEventListener('submit', function(e) {
        e.preventDefault();
        const formData = new FormData(this);
        const msgEl = document.getElementById('loginMessage');
        msgEl.textContent = '';
        msgEl.className = 'form-message';

        fetch('api/auth.php', {
            method: 'POST',
            body: formData
        })
        .then(r => r.json())
        .then(data => {
            msgEl.className = 'form-message ' + (data.success ? 'success' : 'error');
            msgEl.textContent = data.message || (data.success ? 'Login successful!' : 'Invalid credentials.');
            if (data.success && data.redirect) {
                setTimeout(() => window.location.href = data.redirect, 1000);
            }
        })
        .catch(err => {
            msgEl.className = 'form-message error';
            msgEl.textContent = 'An error occurred. Please try again.';
        });
    });

    signupForm.addEventListener('submit', function(e) {
        e.preventDefault();
        const formData = new FormData(this);
        const msgEl = document.getElementById('signupMessage');
        msgEl.textContent = '';
        msgEl.className = 'form-message';

        // Validate signature has drawing
        const sigData = ctx.getImageData(0, 0, canvas.width, canvas.height);
        let hasDrawing = false;
        for (let i = 0; i < sigData.data.length; i += 4) {
            if (sigData.data[i + 3] > 10) {
                hasDrawing = true;
                break;
            }
        }
        if (!hasDrawing) {
            msgEl.className = 'form-message error';
            msgEl.textContent = 'Please draw your signature above.';
            return;
        }

        formData.set('signature_data', canvas.toDataURL('image/png'));

        const url = (document.body.dataset.debug === '1') ? 'api/auth.php?debug=1' : 'api/auth.php';
        fetch(url, {
            method: 'POST',
            body: formData
        })
        .then(r => {
            return r.text().then(text => {
                try {
                    return JSON.parse(text);
                } catch (e) {
                    throw new Error(text || 'Invalid server response');
                }
            });
        })
        .then(data => {
            msgEl.className = 'form-message ' + (data.success ? 'success' : 'error');
            msgEl.textContent = data.message || (data.success ? 'Signup successful! Redirecting...' : 'Signup failed.');
            if (data.success && data.redirect) {
                setTimeout(() => window.location.href = data.redirect, 1500);
            }
        })
        .catch(err => {
            msgEl.className = 'form-message error';
            const msg = err.message || '';
            msgEl.textContent = (msg.length > 0 && msg.length <= 300) ? msg : 'An error occurred. Please try again.';
        });
    });
});
