<?php
/**
 * Resume Parser - Extracts data from PDF resume and stores in database
 * Uses pure PHP (Smalot PdfParser) - no Python or shell_exec required
 */
class ResumeParser
{
    private $conn;

    public function __construct($dbConnection)
    {
        $this->conn = $dbConnection;
    }

    /**
     * Parse PDF and return structured data array
     */
    public function parsePdf($pdfPath, $first_name = '', $last_name = '')
    {
        $absolutePath = realpath($pdfPath);
        if (!$absolutePath || !file_exists($absolutePath)) {
            return null;
        }

        try {
            $autoload = dirname(__DIR__) . '/vendor/autoload.php';
            if (!file_exists($autoload)) {
                return $this->getFallbackData($first_name, $last_name);
            }
            require_once $autoload;
            $parser = new \Smalot\PdfParser\Parser();
            $pdf = $parser->parseFile($absolutePath);
            $text = $pdf->getText();
        } catch (Throwable $e) {
            return $this->getFallbackData($first_name, $last_name);
        }

        if (empty(trim($text ?? ''))) {
            return $this->getFallbackData($first_name, $last_name);
        }

        return $this->parseResumeText($text, $first_name, $last_name);
    }

    /**
     * Parse resume text into structured data (ported from Python logic)
     */
    private function parseResumeText($text, $first_name, $last_name)
    {
        $result = [
            'full_name' => '',
            'location' => '',
            'headline' => 'Software Engineer | Web Developer',
            'about' => '',
            'about_extended' => '',
            'open_to_work_roles' => 'Software Engineer, Full-Stack Developer, Web Developer',
            'experiences' => [],
            'education' => [],
            'skills' => [],
            'top_skills' => [],
            'certifications' => []
        ];

        $textClean = str_replace("\r", "\n", $text);
        $lines = array_filter(array_map('trim', explode("\n", $textClean)));

        // Find name and location
        $allLines = array_values($lines);
        $lastLines = array_slice($allLines, -15);
        foreach ($lastLines as $i => $line) {
            if (preg_match('/^[A-Z][a-z]+\s+[A-Z][a-z]+$/', $line) && strlen($line) < 40 && strpos($line, ',') === false) {
                $result['full_name'] = $line;
                if (isset($lastLines[$i + 1]) && (strpos($lastLines[$i + 1], ',') !== false || preg_match('/^[A-Za-z\s,]+$/', $lastLines[$i + 1]))) {
                    $result['location'] = $lastLines[$i + 1];
                }
                break;
            }
        }
        if (empty($result['full_name']) && ($first_name || $last_name)) {
            $result['full_name'] = trim("$first_name $last_name");
        }

        $getSection = function ($content, $startMarker, $endMarkers) {
            $idx = stripos($content, $startMarker);
            if ($idx === false) return '';
            $start = $idx + strlen($startMarker);
            $end = strlen($content);
            foreach ($endMarkers as $em) {
                $eidx = stripos($content, $em, $start);
                if ($eidx !== false) $end = min($end, $eidx);
            }
            return trim(substr($content, $start, $end - $start));
        };

        // Parse SKILLS
        $skillsSection = $getSection($text, 'SKILLS', ['EXPERIENCE', 'EDUCATION', 'WORK', 'EMPLOYMENT']);
        $allSkills = [];
        if ($skillsSection) {
            preg_match_all('/[A-Z][A-Z\s&]+[|:]\s*([^\n]+)/', $skillsSection, $skillItems);
            foreach ($skillItems[1] ?? [] as $s) {
                $parts = preg_split('/[•\|,\s]+/', $s);
                foreach ($parts as $p) {
                    $p = trim($p);
                    if (strlen($p) > 1 && strlen($p) < 50) $allSkills[] = $p;
                }
            }
            preg_match_all('/\b(?:PHP|JavaScript|Python|ReactJS?|NodeJS?|HTML|CSS|MySQL|MongoDB|Firebase|Flutter|WordPress|Figma|Adobe|Bootstrap|Redux|Express|Agile|REST|MVC)\b/i', $text, $techWords);
            $allSkills = array_values(array_unique(array_merge($allSkills, $techWords[0] ?? [])));
            $allSkills = array_slice($allSkills, 0, 20);
            $result['top_skills'] = array_slice($allSkills, 0, 5);
            foreach (array_slice($allSkills, 0, 15) as $i => $sk) {
                $result['skills'][] = [
                    'skill_name' => $sk,
                    'institution' => '',
                    'is_top_skill' => in_array($sk, $result['top_skills']) ? 1 : 0
                ];
            }
        }

        if (!empty($result['top_skills'])) {
            $result['headline'] = implode(' | ', array_slice($result['top_skills'], 0, 8));
        }

        // Parse EXPERIENCE
        $expSection = $getSection($text, 'EXPERIENCE', ['EDUCATION', 'COURSEWORK', 'VOLUNTEER', 'CERTIFICATES']);
        if ($expSection) {
            $expBlocks = preg_split('/\n(?=[A-Z][a-zA-Z\s&]+(?:\s+\||\s+·))/', $expSection);
            foreach ($expBlocks as $block) {
                $block = trim($block);
                if (strlen($block) < 20) continue;
                $linesBlock = array_values(array_filter(array_map('trim', explode("\n", $block))));
                if (count($linesBlock) >= 2) {
                    $firstLine = $linesBlock[0];
                    $secondLine = $linesBlock[1];
                    $company = strpos($firstLine, '|') !== false ? trim(explode('|', $firstLine, 2)[0]) : $firstLine;
                    $skillsStr = strpos($firstLine, '|') !== false ? trim(explode('|', $firstLine, 2)[1] ?? '') : '';
                    $jobTitle = $secondLine;
                    $dateLocation = '';
                    if (preg_match('/\s+[|·]\s+/', $secondLine)) {
                        $parts = preg_split('/\s+[|·]\s+/', $secondLine, 2);
                        $jobTitle = trim($parts[0]);
                        $dateLocation = trim($parts[1] ?? '');
                    }
                    preg_match('/([A-Za-z]{3,9}\s+\d{4})\s*[–\-]\s*([A-Za-z]{3,9}\s+\d{4}|Present|[\w\s]+)/', $dateLocation, $dateMatch);
                    $startDate = $dateMatch[1] ?? '';
                    $endDate = $dateMatch[2] ?? '';
                    preg_match('/·\s*(\d+\s*mos?)/', $dateLocation, $durationMatch);
                    $duration = $durationMatch[1] ?? '';
                    $location = trim(preg_replace('/[A-Za-z]{3}\s+\d{4}.*$/', '', $dateLocation));
                    $desc = count($linesBlock) > 2 ? substr(implode("\n", array_slice($linesBlock, 2)), 0, 500) : '';
                    $employmentType = (stripos($jobTitle, 'intern') !== false || stripos($company, 'intern') !== false) ? 'Internship' : 'Full-time';
                    $result['experiences'][] = [
                        'company_name' => substr($company, 0, 255),
                        'job_title' => substr($jobTitle, 0, 255),
                        'employment_type' => $employmentType,
                        'start_date' => $startDate,
                        'end_date' => $endDate,
                        'duration' => $duration,
                        'location' => substr($location, 0, 255),
                        'skills' => substr($skillsStr, 0, 500),
                        'description' => $desc
                    ];
                }
            }
        }

        // Parse EDUCATION
        $eduSection = $getSection($text, 'EDUCATION', ['COURSEWORK', 'VOLUNTEER', 'CERTIFICATES', 'EXPERIENCE', 'SKILLS']);
        if ($eduSection) {
            $eduBlocks = preg_split('/\n(?=[A-Z][a-zA-Z\s&()]+(?:University|College|School|Institute))/', $eduSection);
            foreach ($eduBlocks as $block) {
                $block = trim($block);
                if (strlen($block) < 15) continue;
                $linesBlock = array_values(array_filter(array_map('trim', explode("\n", $block))));
                if (!empty($linesBlock)) {
                    $institution = $linesBlock[0];
                    $degree = '';
                    $dates = '';
                    $grade = '';
                    $locationEdu = '';
                    $skillsEdu = '';
                    foreach (array_slice($linesBlock, 1) as $line) {
                        if (preg_match('/^[A-Za-z\.\s,]+,/', $line) || preg_match('/B\.?S\.?|Bachelors|I\.?C\.?S|M\.?S/', $line)) {
                            $degree = $line;
                        } elseif (preg_match('/[A-Za-z]{3}\s+\d{4}/', $line)) {
                            $dates = strpos($line, '|') !== false ? trim(explode('|', $line)[0]) : $line;
                            if (strpos($line, '|') !== false) {
                                $parts = explode('|', $line);
                                $locationEdu = trim(end($parts));
                            }
                        } elseif (preg_match('/(?:CGPA|Grade|GPA)[:\s]*/i', $line)) {
                            $grade = $line;
                        } elseif (strpos($line, '·') !== false || strpos($line, '|') !== false) {
                            $skillsEdu = $line;
                        }
                    }
                    $result['education'][] = [
                        'institution_name' => substr($institution, 0, 255),
                        'degree' => substr($degree ?: 'N/A', 0, 255),
                        'start_date' => '',
                        'end_date' => $dates,
                        'grade' => substr($grade, 0, 100),
                        'location' => substr($locationEdu, 0, 255),
                        'skills' => substr($skillsEdu, 0, 500)
                    ];
                }
            }
        }

        // Parse CERTIFICATES
        $certSection = $getSection($text, 'CERTIFICATES', ['INTERESTS', 'EDUCATION', 'SKILLS']) ?: $getSection($text, 'CERTIFICATIONS', ['INTERESTS', 'EDUCATION']);
        if ($certSection) {
            $certLines = array_filter(array_map('trim', explode("\n", $certSection)), function ($l) { return strlen($l) > 5; });
            $certLines = array_values($certLines);
            $i = 0;
            while ($i < count($certLines)) {
                $title = $certLines[$i];
                $issuer = '';
                $date = '';
                if (isset($certLines[$i + 1]) && (!preg_match('/^[A-Z][a-z]/', $certLines[$i + 1]) || strlen($certLines[$i + 1]) < 30)) {
                    $issuer = $certLines[$i + 1];
                    $i++;
                }
                if (isset($certLines[$i + 1]) && preg_match('/[A-Za-z]{3}\s+\d{4}/', $certLines[$i + 1])) {
                    $date = $certLines[$i + 1];
                    $i++;
                }
                $i++;
                if (strlen($title) > 3) {
                    $result['certifications'][] = [
                        'title' => substr($title, 0, 500),
                        'issuer' => substr($issuer, 0, 255),
                        'issued_date' => substr($date, 0, 100),
                        'skills' => ''
                    ];
                }
            }
        }

        // Build about from skills + education
        if (empty($result['about']) && (!empty($result['education']) || !empty($result['skills']))) {
            $skillStr = !empty($result['top_skills']) ? implode(', ', array_slice($result['top_skills'], 0, 8)) : 'full-stack development';
            $result['about'] = "Software Engineering graduate with experience in $skillStr. Skilled in building web applications with clean architecture and user-focused design.";
            $result['about_extended'] = "Experienced in RESTful APIs, MVC, and Agile methodologies. Proficient with modern development tools. Strong soft skills in leadership, communication, and problem-solving.";
        }

        return $result;
    }

    /**
     * Store parsed resume data in database for user
     */
    public function storeInDatabase($userId, $parsedData, $formData = [])
    {
        $userId = (int) $userId;
        if ($userId <= 0) return false;

        $first_name = $formData['first_name'] ?? '';
        $last_name = $formData['last_name'] ?? '';
        $full_name = $parsedData['full_name'] ?? trim("$first_name $last_name");
        if (empty($full_name)) {
            $full_name = trim("$first_name $last_name");
        }

        $headline = $parsedData['headline'] ?? 'Software Engineer | Web Developer';
        $location = $parsedData['location'] ?? '';
        $about = $parsedData['about'] ?? '';
        $about_extended = $parsedData['about_extended'] ?? '';
        $open_to_work = $parsedData['open_to_work_roles'] ?? 'Software Engineer, Web Developer';

        $this->updateUserProfile($userId, [
            'headline' => $headline,
            'location' => $location,
            'about' => $about,
            'about_extended' => $about_extended,
            'open_to_work_roles' => $open_to_work
        ]);

        $this->storeExperiences($userId, $parsedData['experiences'] ?? []);
        $this->storeEducation($userId, $parsedData['education'] ?? []);
        $this->storeSkills($userId, $parsedData['skills'] ?? [], $parsedData['top_skills'] ?? []);
        $this->storeCertifications($userId, $parsedData['certifications'] ?? []);

        return true;
    }

    private function updateUserProfile($userId, $fields)
    {
        $allowed = ['headline', 'location', 'about', 'about_extended', 'open_to_work_roles'];
        $existing = [];
        $r = $this->conn->query("SHOW COLUMNS FROM users");
        if ($r) {
            while ($row = $r->fetch_assoc()) {
                $existing[] = $row['Field'];
            }
        }
        $updates = [];
        $types = '';
        $params = [];
        foreach ($fields as $key => $val) {
            if (in_array($key, $allowed) && in_array($key, $existing)) {
                $updates[] = "`$key` = ?";
                $types .= 's';
                $params[] = $val;
            }
        }
        if (empty($updates)) return;
        $params[] = $userId;
        $types .= 'i';
        $sql = "UPDATE users SET " . implode(', ', $updates) . " WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        if (!$stmt) return;
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
    }

    private function storeExperiences($userId, $experiences)
    {
        if (!$this->tableExists('user_experiences')) return;
        $stmt = $this->conn->prepare("DELETE FROM user_experiences WHERE user_id = ?");
        if (!$stmt) return;
        $stmt->bind_param('i', $userId);
        $stmt->execute();

        $insert = $this->conn->prepare("INSERT INTO user_experiences (user_id, company_name, job_title, employment_type, start_date, end_date, duration, location, skills, description, display_order) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        if (!$insert) return;
        foreach ($experiences as $i => $exp) {
            $companyName = $exp['company_name'] ?? '';
            $jobTitle = $exp['job_title'] ?? '';
            $employmentType = $exp['employment_type'] ?? 'Full-time';
            $startDate = $exp['start_date'] ?? '';
            $endDate = $exp['end_date'] ?? '';
            $duration = $exp['duration'] ?? '';
            $location = $exp['location'] ?? '';
            $skills = $exp['skills'] ?? '';
            $description = $exp['description'] ?? '';
            $displayOrder = $i;
            
            $insert->bind_param('isssssssssi',
                $userId,
                $companyName,
                $jobTitle,
                $employmentType,
                $startDate,
                $endDate,
                $duration,
                $location,
                $skills,
                $description,
                $displayOrder
            );
            $insert->execute();
        }
    }

    private function storeEducation($userId, $education)
    {
        if (!$this->tableExists('user_education')) return;
        $stmt = $this->conn->prepare("DELETE FROM user_education WHERE user_id = ?");
        if (!$stmt) return;
        $stmt->bind_param('i', $userId);
        $stmt->execute();

        $insert = $this->conn->prepare("INSERT INTO user_education (user_id, institution_name, degree, start_date, end_date, grade, location, skills, display_order) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        if (!$insert) return;
        foreach ($education as $i => $edu) {
            $institutionName = $edu['institution_name'] ?? '';
            $degree = $edu['degree'] ?? '';
            $startDate = $edu['start_date'] ?? '';
            $endDate = $edu['end_date'] ?? '';
            $grade = $edu['grade'] ?? '';
            $location = $edu['location'] ?? '';
            $skills = $edu['skills'] ?? '';
            $displayOrder = $i;
            
            $insert->bind_param('isssssssi',
                $userId,
                $institutionName,
                $degree,
                $startDate,
                $endDate,
                $grade,
                $location,
                $skills,
                $displayOrder
            );
            $insert->execute();
        }
    }

    private function storeSkills($userId, $skills, $topSkills = [])
    {
        if (!$this->tableExists('user_skills')) return;
        $stmt = $this->conn->prepare("DELETE FROM user_skills WHERE user_id = ?");
        if (!$stmt) return;
        $stmt->bind_param('i', $userId);
        $stmt->execute();

        if (empty($skills) && !empty($topSkills)) {
            foreach ($topSkills as $sk) {
                $skills[] = ['skill_name' => $sk, 'institution' => '', 'is_top_skill' => 1];
            }
        }

        $insert = $this->conn->prepare("INSERT INTO user_skills (user_id, skill_name, institution, is_top_skill, display_order) VALUES (?, ?, ?, ?, ?)");
        if (!$insert) return;
        foreach ($skills as $i => $sk) {
            $name = is_array($sk) ? ($sk['skill_name'] ?? $sk) : $sk;
            $inst = is_array($sk) ? ($sk['institution'] ?? '') : '';
            $top = is_array($sk) ? (int)($sk['is_top_skill'] ?? 0) : 0;
            $displayOrder = $i;
            $insert->bind_param('issii', $userId, $name, $inst, $top, $displayOrder);
            $insert->execute();
        }
    }

    private function storeCertifications($userId, $certs)
    {
        if (!$this->tableExists('user_certifications')) return;
        $stmt = $this->conn->prepare("DELETE FROM user_certifications WHERE user_id = ?");
        if (!$stmt) return;
        $stmt->bind_param('i', $userId);
        $stmt->execute();

        $insert = $this->conn->prepare("INSERT INTO user_certifications (user_id, title, issuer, issued_date, credential_id, skills, display_order) VALUES (?, ?, ?, ?, ?, ?, ?)");
        if (!$insert) return;
        foreach ($certs as $i => $cert) {
            $title = $cert['title'] ?? '';
            $issuer = $cert['issuer'] ?? '';
            $issuedDate = $cert['issued_date'] ?? '';
            $credentialId = $cert['credential_id'] ?? '';
            $skills = $cert['skills'] ?? '';
            $displayOrder = $i;
            
            $insert->bind_param('isssssi',
                $userId,
                $title,
                $issuer,
                $issuedDate,
                $credentialId,
                $skills,
                $displayOrder
            );
            $insert->execute();
        }
    }

    private function tableExists($table)
    {
        $r = $this->conn->query("SHOW TABLES LIKE '$table'");
        return $r && $r->num_rows > 0;
    }

    private function getFallbackData($first_name, $last_name)
    {
        return [
            'full_name' => trim("$first_name $last_name"),
            'location' => '',
            'headline' => 'Software Engineer | Web Developer',
            'about' => '',
            'about_extended' => '',
            'open_to_work_roles' => 'Software Engineer, Web Developer',
            'experiences' => [],
            'education' => [],
            'skills' => [],
            'top_skills' => [],
            'certifications' => []
        ];
    }
}
