/**
 * Premium Landing Page - GSAP Animations
 * Using ScrollTrigger for smooth reveal animations
 */

// Register ScrollTrigger plugin
gsap.registerPlugin(ScrollTrigger);

// Wait for DOM to be ready
document.addEventListener('DOMContentLoaded', function() {
    
    // Fallback: Ensure all animated content becomes visible after 2 seconds if animations haven't played
    setTimeout(function() {
        const animatedElements = document.querySelectorAll('[data-animate], .section-heading, .section-body, .feature-card, .glass-card');
        animatedElements.forEach(function(el) {
            if (window.getComputedStyle(el).opacity === '0' || window.getComputedStyle(el).opacity < 0.1) {
                gsap.to(el, { opacity: 1, y: 0, scale: 1, duration: 0.5, ease: 'power2.out' });
            }
        });
    }, 2000);
    
    // ============================================
    // Navbar scroll effect
    // ============================================
    const navbar = document.getElementById('mainNav');
    let lastScroll = 0;
    
    window.addEventListener('scroll', function() {
        const currentScroll = window.pageYOffset;
        
        if (currentScroll > 100) {
            navbar.classList.add('scrolled');
        } else {
            navbar.classList.remove('scrolled');
        }
        
        lastScroll = currentScroll;
    });

    // ============================================
    // Hero Section Animations (Stagger)
    // ============================================
    const heroElements = gsap.utils.toArray('[data-animate="hero"]');
    
    // Set initial state
    gsap.set(heroElements, { opacity: 0, y: 50 });
    
    // Animate in
    gsap.to(heroElements, {
        opacity: 1,
        y: 0,
        duration: 0.8,
        stagger: 0.1,
        ease: 'power3.out',
        scrollTrigger: {
            trigger: '#hero',
            start: 'top 80%',
            toggleActions: 'play none none none',
            once: true
        }
    });

    // ============================================
    // Section Headings & Body Text (Fade Up)
    // ============================================
    const sectionHeadings = gsap.utils.toArray('.section-heading');
    const sectionBodies = gsap.utils.toArray('.section-body');
    
    sectionHeadings.forEach((heading) => {
        const rect = heading.getBoundingClientRect();
        const isInViewport = rect.top < window.innerHeight * 0.85;
        
        if (isInViewport) {
            // If already visible, show immediately
            gsap.set(heading, { opacity: 1, y: 0 });
        } else {
            // Use fromTo for scroll-triggered animation
            gsap.fromTo(heading,
                { opacity: 0, y: 40 },
                {
                    opacity: 1,
                    y: 0,
                    duration: 0.6,
                    ease: 'power2.out',
                    scrollTrigger: {
                        trigger: heading,
                        start: 'top 90%',
                        toggleActions: 'play none none none',
                        once: true
                    }
                }
            );
        }
    });
    
    sectionBodies.forEach((body) => {
        const rect = body.getBoundingClientRect();
        const isInViewport = rect.top < window.innerHeight * 0.85;
        
        if (isInViewport) {
            // If already visible, show immediately
            gsap.set(body, { opacity: 1, y: 0 });
        } else {
            // Use fromTo for scroll-triggered animation
            gsap.fromTo(body,
                { opacity: 0, y: 40 },
                {
                    opacity: 1,
                    y: 0,
                    duration: 0.8,
                    delay: 0.2,
                    ease: 'power3.out',
                    scrollTrigger: {
                        trigger: body,
                        start: 'top 85%',
                        toggleActions: 'play none none none',
                        once: true
                    }
                }
            );
        }
    });

    // ============================================
    // Feature Cards Stagger Animation
    // ============================================
    const featureCards = gsap.utils.toArray('.feature-card');
    const curiositySection = document.getElementById('curiosity');
    
    if (featureCards.length > 0 && curiositySection) {
        // Check if section is already in viewport
        const rect = curiositySection.getBoundingClientRect();
        const isInViewport = rect.top < window.innerHeight * 0.75;
        
        if (isInViewport) {
            // If already visible, animate immediately
            gsap.to(featureCards, {
                opacity: 1,
                y: 0,
                scale: 1,
                duration: 0.6,
                stagger: 0.1,
                ease: 'power2.out'
            });
        } else {
            // Use fromTo for scroll-triggered animation
            gsap.fromTo(featureCards, 
                {
                    opacity: 0,
                    y: 50,
                    scale: 0.9
                },
                {
                    opacity: 1,
                    y: 0,
                    scale: 1,
                    duration: 0.8,
                    stagger: 0.15,
                    ease: 'power3.out',
                    scrollTrigger: {
                        trigger: '#curiosity',
                        start: 'top 75%',
                        toggleActions: 'play none none none',
                        once: true
                    }
                }
            );
        }
    }

    // ============================================
    // Glass Cards Fade In
    // ============================================
    const glassCards = gsap.utils.toArray('.glass-card:not(.feature-card)');
    
    glassCards.forEach((card) => {
        // Set initial state
        gsap.set(card, { opacity: 0, y: 40 });
        
        // Animate in
        gsap.to(card, {
            opacity: 1,
            y: 0,
            duration: 1,
            ease: 'power3.out',
            scrollTrigger: {
                trigger: card,
                start: 'top 85%',
                toggleActions: 'play none none none',
                once: true
            }
        });
    });

    // ============================================
    // Closing CTA Pulse Animation
    // ============================================
    const closingCTA = document.querySelector('.closing-cta');
    
    if (closingCTA) {
        // Set initial state
        gsap.set(closingCTA, { opacity: 0, scale: 0.8 });
        
        // Animate in
        gsap.to(closingCTA, {
            opacity: 1,
            scale: 1,
            duration: 0.8,
            ease: 'back.out(1.7)',
            scrollTrigger: {
                trigger: closingCTA,
                start: 'top 85%',
                toggleActions: 'play none none none',
                once: true
            }
        });

        // Continuous subtle pulse
        gsap.to(closingCTA, {
            scale: 1.05,
            duration: 1.5,
            repeat: -1,
            yoyo: true,
            ease: 'power1.inOut',
            delay: 1
        });
    }

    // ============================================
    // Smooth Scroll for Anchor Links
    // ============================================
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            
            if (target) {
                const offsetTop = target.offsetTop - 80; // Account for navbar
                window.scrollTo({
                    top: offsetTop,
                    behavior: 'smooth'
                });
            }
        });
    });

    // ============================================
    // Button Hover Microinteractions
    // ============================================
    const buttons = document.querySelectorAll('.btn-primary-cta');
    
    buttons.forEach(button => {
        button.addEventListener('mouseenter', function() {
            gsap.to(button, {
                scale: 1.05,
                duration: 0.3,
                ease: 'power2.out'
            });
        });
        
        button.addEventListener('mouseleave', function() {
            gsap.to(button, {
                scale: 1,
                duration: 0.3,
                ease: 'power2.out'
            });
        });
    });

    // ============================================
    // Service Cards Interactive Animations
    // ============================================
    const serviceCards = document.querySelectorAll('.service-card-modern');
    
    serviceCards.forEach((card, index) => {
        // Magnetic hover effect
        card.addEventListener('mousemove', function(e) {
            const rect = card.getBoundingClientRect();
            const x = e.clientX - rect.left;
            const y = e.clientY - rect.top;
            
            const centerX = rect.width / 2;
            const centerY = rect.height / 2;
            
            const moveX = (x - centerX) / 15;
            const moveY = (y - centerY) / 15;
            
            gsap.to(card, {
                x: moveX,
                y: moveY - 12,
                duration: 0.3,
                ease: 'power2.out'
            });
        });
        
        card.addEventListener('mouseleave', function() {
            gsap.to(card, {
                x: 0,
                y: 0,
                duration: 0.5,
                ease: 'power2.out'
            });
        });
        
        // 3D Tilt effect
        card.addEventListener('mousemove', function(e) {
            const rect = card.getBoundingClientRect();
            const x = e.clientX - rect.left;
            const y = e.clientY - rect.top;
            
            const centerX = rect.width / 2;
            const centerY = rect.height / 2;
            
            const rotateX = (y - centerY) / 20;
            const rotateY = (centerX - x) / 20;
            
            gsap.to(card, {
                rotationX: rotateX,
                rotationY: rotateY,
                transformPerspective: 1000,
                duration: 0.3,
                ease: 'power2.out'
            });
        });
        
        card.addEventListener('mouseleave', function() {
            gsap.to(card, {
                rotationX: 0,
                rotationY: 0,
                duration: 0.5,
                ease: 'power2.out'
            });
        });
        
        // Ripple effect on click
        card.addEventListener('click', function(e) {
            const ripple = document.createElement('span');
            const rect = card.getBoundingClientRect();
            const size = Math.max(rect.width, rect.height);
            const x = e.clientX - rect.left - size / 2;
            const y = e.clientY - rect.top - size / 2;
            
            ripple.style.width = ripple.style.height = size + 'px';
            ripple.style.left = x + 'px';
            ripple.style.top = y + 'px';
            ripple.classList.add('service-ripple');
            
            card.appendChild(ripple);
            
            setTimeout(() => {
                ripple.remove();
            }, 600);
        });
        
        // Stagger animation on scroll
        gsap.fromTo(card,
            {
                opacity: 0,
                y: 60,
                scale: 0.9,
                rotationX: -15
            },
            {
                opacity: 1,
                y: 0,
                scale: 1,
                rotationX: 0,
                duration: 0.8,
                delay: index * 0.1,
                ease: 'power3.out',
                scrollTrigger: {
                    trigger: card,
                    start: 'top 85%',
                    toggleActions: 'play none none none',
                    once: true
                }
            }
        );
        
        // Icon animation on hover
        const icon = card.querySelector('.service-icon');
        if (icon) {
            card.addEventListener('mouseenter', function() {
                gsap.to(icon, {
                    scale: 1.2,
                    rotation: 5,
                    y: -5,
                    duration: 0.5,
                    ease: 'back.out(1.7)'
                });
            });
            
            card.addEventListener('mouseleave', function() {
                gsap.to(icon, {
                    scale: 1,
                    rotation: 0,
                    y: 0,
                    duration: 0.4,
                    ease: 'power2.out'
                });
            });
        }
        
        // Mouse-following glow effect
        const glowEffect = document.createElement('div');
        glowEffect.style.cssText = `
            position: absolute;
            width: 200px;
            height: 200px;
            background: radial-gradient(circle, rgba(96, 165, 250, 0.3) 0%, transparent 70%);
            border-radius: 50%;
            pointer-events: none;
            opacity: 0;
            transform: translate(-50%, -50%);
            transition: opacity 0.3s ease;
            z-index: 1;
        `;
        card.style.position = 'relative';
        card.appendChild(glowEffect);
        
        card.addEventListener('mousemove', function(e) {
            const rect = card.getBoundingClientRect();
            const x = e.clientX - rect.left;
            const y = e.clientY - rect.top;
            
            glowEffect.style.left = x + 'px';
            glowEffect.style.top = y + 'px';
            glowEffect.style.opacity = '1';
        });
        
        card.addEventListener('mouseleave', function() {
            glowEffect.style.opacity = '0';
        });
        
        // Text reveal animation on hover
        const title = card.querySelector('.service-title');
        const tagline = card.querySelector('.service-tagline');
        const description = card.querySelector('.service-description');
        
        card.addEventListener('mouseenter', function() {
            if (title) {
                gsap.fromTo(title,
                    { x: -20, opacity: 0.8 },
                    { x: 0, opacity: 1, duration: 0.4, ease: 'power2.out' }
                );
            }
            if (tagline) {
                gsap.fromTo(tagline,
                    { x: -15, opacity: 0.7 },
                    { x: 0, opacity: 1, duration: 0.5, delay: 0.1, ease: 'power2.out' }
                );
            }
            if (description) {
                gsap.fromTo(description,
                    { x: -10, opacity: 0.8 },
                    { x: 0, opacity: 1, duration: 0.5, delay: 0.2, ease: 'power2.out' }
                );
            }
        });
        
        // Parallax effect on scroll
        gsap.to(card, {
            y: -20 * (index % 2 === 0 ? 1 : -1),
            scrollTrigger: {
                trigger: card,
                start: 'top bottom',
                end: 'bottom top',
                scrub: 1
            }
        });
    });
    
    // ============================================
    // Card Hover Microinteractions (for other cards)
    // ============================================
    const cards = document.querySelectorAll('.glass-card:not(.service-card-modern)');
    
    cards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            gsap.to(card, {
                y: -5,
                duration: 0.3,
                ease: 'power2.out'
            });
        });
        
        card.addEventListener('mouseleave', function() {
            gsap.to(card, {
                y: 0,
                duration: 0.3,
                ease: 'power2.out'
            });
        });
    });

    // ============================================
    // Parallax Effect for Hero Image
    // ============================================
    const heroImage = document.querySelector('.hero-image-wrapper');
    
    if (heroImage) {
        gsap.to(heroImage, {
            y: -50,
            scrollTrigger: {
                trigger: '#hero',
                start: 'top top',
                end: 'bottom top',
                scrub: 1
            }
        });
    }

    // ============================================
    // Animated Background Blobs
    // ============================================
    const blobs = document.querySelectorAll('.blob');
    
    blobs.forEach((blob, index) => {
        gsap.to(blob, {
            x: 'random(-100, 100)',
            y: 'random(-100, 100)',
            duration: 'random(15, 25)',
            repeat: -1,
            yoyo: true,
            ease: 'power1.inOut',
            delay: index * 2
        });
    });

    // ============================================
    // Cards Slider Navigation
    // ============================================
    const sliderTrack = document.querySelector('.slider-track');
    const sliderCards = document.querySelectorAll('.slider-card');
    const sliderCardWrappers = document.querySelectorAll('.slider-card-wrapper');
    const prevBtn = document.querySelector('.slider-nav-prev');
    const nextBtn = document.querySelector('.slider-nav-next');
    const indicators = document.querySelectorAll('.slider-indicator');
    const progressBar = document.querySelector('.slider-progress-bar');
    const sliderWrapper = document.querySelector('.cards-slider-wrapper');
    let currentCardIndex = 0;
    let autoPlayInterval = null;
    let isPaused = false;

    if (sliderTrack && sliderCards.length > 0 && prevBtn && nextBtn) {
        const totalCards = sliderCards.length;
        const AUTO_PLAY_DURATION = 5000; // 5 seconds

        // Update slider position with animation
        function updateSliderPosition(animate = true) {
            const translateX = -currentCardIndex * 100;
            
            if (animate) {
                gsap.to(sliderTrack, {
                    x: translateX + '%',
                    duration: 0.6,
                    ease: 'power3.out'
                });
            } else {
                gsap.set(sliderTrack, { x: translateX + '%' });
            }

            // Update indicators
            indicators.forEach((indicator, index) => {
                if (index === currentCardIndex) {
                    indicator.classList.add('active');
                } else {
                    indicator.classList.remove('active');
                }
            });

            // Update active card wrapper
            sliderCardWrappers.forEach((wrapper, index) => {
                if (index === currentCardIndex) {
                    wrapper.classList.add('active');
                    // Animate card entrance
                    gsap.fromTo(wrapper.querySelector('.slider-card'),
                        { opacity: 0.8, y: 20 },
                        { opacity: 1, y: 0, duration: 0.5, ease: 'power2.out' }
                    );
                } else {
                    wrapper.classList.remove('active');
                }
            });

            // Update progress bar
            if (progressBar) {
                const progress = ((currentCardIndex + 1) / totalCards) * 100;
                gsap.to(progressBar, {
                    width: progress + '%',
                    duration: 0.3,
                    ease: 'power2.out'
                });
            }
        }

        // Go to specific slide
        function goToSlide(index) {
            if (index >= 0 && index < totalCards) {
                currentCardIndex = index;
                updateSliderPosition();
                resetAutoPlay();
            }
        }

        // Previous button
        prevBtn.addEventListener('click', function() {
            currentCardIndex = (currentCardIndex - 1 + totalCards) % totalCards;
            updateSliderPosition();
            resetAutoPlay();
        });

        // Next button
        nextBtn.addEventListener('click', function() {
            currentCardIndex = (currentCardIndex + 1) % totalCards;
            updateSliderPosition();
            resetAutoPlay();
        });

        // Indicator clicks
        indicators.forEach((indicator, index) => {
            indicator.addEventListener('click', function() {
                goToSlide(index);
            });
        });

        // Card click navigation
        sliderCards.forEach((card, index) => {
            card.addEventListener('click', function(e) {
                // Don't navigate if clicking on a link or button inside
                if (e.target.tagName === 'A' || e.target.tagName === 'BUTTON') {
                    return;
                }
                goToSlide(index);
            });
        });

        // Auto-play functionality
        function startAutoPlay() {
            if (autoPlayInterval) clearInterval(autoPlayInterval);
            autoPlayInterval = setInterval(function() {
                if (!isPaused) {
                    currentCardIndex = (currentCardIndex + 1) % totalCards;
                    updateSliderPosition();
                }
            }, AUTO_PLAY_DURATION);
        }

        function resetAutoPlay() {
            if (autoPlayInterval) clearInterval(autoPlayInterval);
            startAutoPlay();
        }

        function pauseAutoPlay() {
            isPaused = true;
        }

        function resumeAutoPlay() {
            isPaused = false;
        }

        // Pause on hover
        if (sliderWrapper) {
            sliderWrapper.addEventListener('mouseenter', pauseAutoPlay);
            sliderWrapper.addEventListener('mouseleave', resumeAutoPlay);
        }

        // Touch/swipe support for mobile
        let touchStartX = 0;
        let touchEndX = 0;
        let isDragging = false;
        const sliderContainer = document.querySelector('.slider-container');

        if (sliderContainer) {
            sliderContainer.addEventListener('touchstart', function(e) {
                touchStartX = e.changedTouches[0].screenX;
                isDragging = true;
                pauseAutoPlay();
            }, { passive: true });

            sliderContainer.addEventListener('touchmove', function(e) {
                if (isDragging) {
                    touchEndX = e.changedTouches[0].screenX;
                }
            }, { passive: true });

            sliderContainer.addEventListener('touchend', function(e) {
                if (isDragging) {
                    handleSwipe();
                    isDragging = false;
                    resumeAutoPlay();
                }
            }, { passive: true });
        }

        function handleSwipe() {
            const swipeThreshold = 50;
            const diff = touchStartX - touchEndX;

            if (Math.abs(diff) > swipeThreshold) {
                if (diff > 0) {
                    // Swipe left - next
                    nextBtn.click();
                } else {
                    // Swipe right - previous
                    prevBtn.click();
                }
            }
        }

        // Keyboard navigation
        document.addEventListener('keydown', function(e) {
            // Only navigate if slider is in viewport
            const sliderSection = document.getElementById('cards-slider');
            if (sliderSection) {
                const rect = sliderSection.getBoundingClientRect();
                const isInView = rect.top < window.innerHeight && rect.bottom > 0;
                
                if (isInView) {
                    if (e.key === 'ArrowLeft') {
                        e.preventDefault();
                        prevBtn.click();
                    } else if (e.key === 'ArrowRight') {
                        e.preventDefault();
                        nextBtn.click();
                    }
                }
            }
        });

        // Initialize
        updateSliderPosition(false);
        startAutoPlay();

        // Add ripple effect on card click
        sliderCards.forEach(card => {
            card.addEventListener('click', function(e) {
                const ripple = document.createElement('span');
                const rect = card.getBoundingClientRect();
                const size = Math.max(rect.width, rect.height);
                const x = e.clientX - rect.left - size / 2;
                const y = e.clientY - rect.top - size / 2;
                
                ripple.style.width = ripple.style.height = size + 'px';
                ripple.style.left = x + 'px';
                ripple.style.top = y + 'px';
                ripple.classList.add('ripple-effect');
                
                card.appendChild(ripple);
                
                setTimeout(() => {
                    ripple.remove();
                }, 600);
            });
        });
    }

    // ============================================
    // Refresh ScrollTrigger on resize
    // ============================================
    let resizeTimer;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(function() {
            ScrollTrigger.refresh();
        }, 250);
    });

});
