<?php
require_once 'config.php';

// Check if PDF parser library is available
$parserAvailable = file_exists('vendor/autoload.php');

if ($parserAvailable) {
    require_once 'vendor/autoload.php';
}

/**
 * Extract text from PDF using smalot/pdfparser
 */
function extractTextFromPDF($pdfPath) {
    try {
        // Check file size - skip files larger than 10MB
        $fileSize = filesize($pdfPath);
        if ($fileSize > 10 * 1024 * 1024) {
            return null; // Skip very large files
        }
        
        // Increase memory limit temporarily for PDF processing
        $oldMemoryLimit = ini_get('memory_limit');
        ini_set('memory_limit', '256M');
        
        // Suppress deprecation warnings from PDF parser
        $oldErrorReporting = error_reporting();
        error_reporting(E_ERROR | E_PARSE);
        
        $parser = new \Smalot\PdfParser\Parser();
        $pdf = $parser->parseFile($pdfPath);
        $text = $pdf->getText();
        
        // Restore settings
        error_reporting($oldErrorReporting);
        ini_set('memory_limit', $oldMemoryLimit);
        
        return $text;
    } catch (Exception $e) {
        // Restore settings in case of exception
        if (isset($oldErrorReporting)) {
            error_reporting($oldErrorReporting);
        }
        if (isset($oldMemoryLimit)) {
            ini_set('memory_limit', $oldMemoryLimit);
        }
        return null;
    } catch (Error $e) {
        // Catch fatal errors like memory exhaustion
        if (isset($oldErrorReporting)) {
            error_reporting($oldErrorReporting);
        }
        if (isset($oldMemoryLimit)) {
            ini_set('memory_limit', $oldMemoryLimit);
        }
        return null;
    }
}

/**
 * Common skills to look for in resumes
 */
function getCommonSkills() {
    return [
        // Programming Languages
        'PHP', 'JavaScript', 'Python', 'Java', 'C++', 'C#', 'Ruby', 'Go', 'Swift', 'Kotlin',
        'TypeScript', 'Rust', 'Scala', 'Perl', 'R', 'Dart', 'Objective-C',
        
        // Web Technologies
        'HTML', 'HTML5', 'CSS', 'CSS3', 'SASS', 'SCSS', 'LESS', 'Bootstrap', 'Tailwind',
        'React', 'Angular', 'Vue.js', 'Vue', 'Next.js', 'Nuxt.js', 'jQuery', 'AJAX',
        'Node.js', 'Express.js', 'Laravel', 'Django', 'Flask', 'Spring', 'ASP.NET',
        
        // Databases
        'MySQL', 'PostgreSQL', 'MongoDB', 'Redis', 'SQLite', 'Oracle', 'SQL Server',
        'MariaDB', 'Cassandra', 'DynamoDB', 'Firebase', 'Elasticsearch',
        
        // DevOps & Tools
        'Git', 'GitHub', 'GitLab', 'Docker', 'Kubernetes', 'Jenkins', 'CI/CD',
        'AWS', 'Azure', 'Google Cloud', 'GCP', 'Linux', 'Unix', 'Bash',
        'Terraform', 'Ansible', 'Nginx', 'Apache',
        
        // Mobile Development
        'Android', 'iOS', 'React Native', 'Flutter', 'Xamarin', 'Ionic',
        
        // Testing & Quality
        'Jest', 'Mocha', 'PHPUnit', 'Selenium', 'JUnit', 'Cypress', 'Testing',
        'Unit Testing', 'Integration Testing', 'TDD', 'BDD',
        
        // Design & UI/UX
        'Photoshop', 'Illustrator', 'Figma', 'Sketch', 'Adobe XD', 'UI/UX',
        'Responsive Design', 'Web Design', 'Graphic Design',
        
        // Other Technologies
        'REST API', 'RESTful', 'GraphQL', 'Microservices', 'API', 'JSON', 'XML',
        'Agile', 'Scrum', 'Jira', 'Webpack', 'Babel', 'NPM', 'Composer',
        'WordPress', 'Shopify', 'Magento', 'WooCommerce',
        
        // Soft Skills
        'Leadership', 'Communication', 'Team Work', 'Problem Solving', 'Project Management',
        'Time Management', 'Critical Thinking', 'Collaboration'
    ];
}

/**
 * Extract skills from text
 */
function extractSkills($text) {
    $commonSkills = getCommonSkills();
    $foundSkills = [];
    
    // Convert text to lowercase for case-insensitive matching
    $textLower = strtolower($text);
    
    foreach ($commonSkills as $skill) {
        $skillLower = strtolower($skill);
        // Use word boundary to avoid partial matches
        if (preg_match('/\b' . preg_quote($skillLower, '/') . '\b/i', $textLower)) {
            $foundSkills[] = $skill;
        }
    }
    
    return array_unique($foundSkills);
}

/**
 * Get all PDFs from the pdfs folder
 */
function getPDFFiles() {
    $pdfDir = 'pdfs';
    $pdfFiles = [];
    
    if (!is_dir($pdfDir)) {
        return $pdfFiles;
    }
    
    $files = scandir($pdfDir);
    foreach ($files as $file) {
        if (pathinfo($file, PATHINFO_EXTENSION) === 'pdf') {
            $pdfFiles[] = [
                'name' => $file,
                'path' => $pdfDir . '/' . $file,
                'size' => filesize($pdfDir . '/' . $file)
            ];
        }
    }
    
    return $pdfFiles;
}

/**
 * Extract candidate name from filename
 * Handles patterns like: Resume_John_Doe.pdf, John_Doe_CV.pdf, John Doe - Title - Resume.pdf
 */
function getCandidateNameFromFilename($filename) {
    $base = pathinfo($filename, PATHINFO_FILENAME);
    
    // If contains " - ", take the part before (e.g. "Rashida Parveen - Web Developer - Resume")
    if (strpos($base, ' - ') !== false) {
        $parts = explode(' - ', $base);
        $base = trim($parts[0]);
    }
    
    // Split camelCase (e.g. RashidAbbasCV -> Rashid Abbas CV)
    $base = preg_replace('/([a-z])([A-Z])/', '$1 $2', $base);
    
    // Words to remove (case-insensitive)
    $removeWords = ['resume', 'cv', 'c.v.', 'application', 'copy', 'final', 'updated', 'new', 
        'compressed', 'v1', 'v2', 'v3', 'pk', 'pro', 'net', 'dev', 'fnl', 'qa', 'sqa',
        'cvresume', 'resumepk', 'job', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
    
    // Replace separators with space
    $base = preg_replace('/[\s_\-\.]+/', ' ', $base);
    $base = preg_replace('/\s+(?:1|2)\s*$/', '', $base); // Remove trailing " 1" or " 2"
    $base = preg_replace('/\d{4}$/', '', $base); // Remove trailing year (2024, 2023)
    $base = trim($base);
    
    // Split into words and filter
    $words = preg_split('/\s+/', $base, -1, PREG_SPLIT_NO_EMPTY);
    $filtered = [];
    foreach ($words as $word) {
        $wordClean = preg_replace('/[^a-zA-Z]/', '', $word);
        if (strlen($wordClean) > 0 && !in_array(strtolower($wordClean), $removeWords) 
            && !in_array(strtolower($word), $removeWords) && !ctype_digit($wordClean)) {
            $filtered[] = ucwords(strtolower($word));
        }
    }
    
    $name = implode(' ', $filtered);
    
    // If we stripped too much, use a simpler fallback
    if (strlen($name) < 3) {
        $base = pathinfo($filename, PATHINFO_FILENAME);
        $base = preg_replace('/[\s_\-\.]+/', ' ', $base);
        $name = ucwords(strtolower(trim($base)));
    }
    
    return $name ?: pathinfo($filename, PATHINFO_FILENAME);
}

/**
 * Format file size
 */
function formatFileSize($bytes) {
    if ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    }
    return $bytes . ' bytes';
}

// Process PDFs
$pdfFiles = getPDFFiles();
$results = [];
$skippedFiles = [];

if ($parserAvailable && !empty($pdfFiles)) {
    foreach ($pdfFiles as $pdf) {
        // Check if file is too large (>10MB)
        if ($pdf['size'] > 10 * 1024 * 1024) {
            $skippedFiles[] = [
                'name' => $pdf['name'],
                'size' => formatFileSize($pdf['size']),
                'reason' => 'File too large (>10MB)'
            ];
            continue;
        }
        
        $text = extractTextFromPDF($pdf['path']);
        $skills = $text ? extractSkills($text) : [];
        
        $results[] = [
            'name' => $pdf['name'],
            'displayName' => getCandidateNameFromFilename($pdf['name']),
            'size' => formatFileSize($pdf['size']),
            'skills' => $skills,
            'skillCount' => count($skills),
            'success' => $text !== null
        ];
    }
}

// Get all unique skills
$allSkills = [];
foreach ($results as $result) {
    $allSkills = array_merge($allSkills, $result['skills']);
}
$allSkills = array_unique($allSkills);
sort($allSkills);

// Sort results alphabetically by candidate display name
usort($results, function($a, $b) {
    $nameA = $a['displayName'] ?? $a['name'];
    $nameB = $b['displayName'] ?? $b['name'];
    return strcasecmp($nameA, $nameB);
});

// A-Z filter (by first letter of candidate display name)
$letterFilter = isset($_GET['letter']) ? strtoupper($_GET['letter']) : '';
if ($letterFilter && strlen($letterFilter) === 1 && ctype_alpha($letterFilter)) {
    $filteredResults = array_filter($results, function($r) use ($letterFilter) {
        $displayName = $r['displayName'] ?? pathinfo($r['name'], PATHINFO_FILENAME);
        $firstChar = strtoupper(substr(trim($displayName), 0, 1));
        return ctype_alpha($firstChar) && $firstChar === $letterFilter;
    });
    $filteredResults = array_values($filteredResults);
} else {
    $letterFilter = '';
    $filteredResults = $results;
}

// Skill filter (resume must have ALL selected skills)
$selectedSkills = [];
if (isset($_GET['skills'])) {
    $selectedSkills = is_array($_GET['skills']) ? $_GET['skills'] : [$_GET['skills']];
    $selectedSkills = array_filter(array_map('trim', $selectedSkills));
}
if (!empty($selectedSkills)) {
    $filteredResults = array_filter($filteredResults, function($r) use ($selectedSkills) {
        $cardSkills = array_map('strtolower', $r['skills']);
        foreach ($selectedSkills as $skill) {
            if (!in_array(strtolower($skill), $cardSkills)) {
                return false;
            }
        }
        return true;
    });
    $filteredResults = array_values($filteredResults);
}

// Pagination: 20 per page
$perPage = 20;
$totalResumes = count($filteredResults);
$totalPages = $totalResumes > 0 ? (int) ceil($totalResumes / $perPage) : 1;
$currentPage = isset($_GET['page']) ? max(1, min((int)$_GET['page'], $totalPages)) : 1;
$offset = ($currentPage - 1) * $perPage;
$paginatedResults = array_slice($filteredResults, $offset, $perPage);

// Build skill filter query string for URLs
function buildSkillParams($selectedSkills) {
    if (empty($selectedSkills)) return '';
    $params = [];
    foreach ($selectedSkills as $s) {
        $params[] = 'skills[]=' . urlencode($s);
    }
    return implode('&', $params);
}
$skillParams = buildSkillParams($selectedSkills);

// Skills to display in filter - only those available in currently filtered resumes
$displaySkills = [];
foreach ($filteredResults as $r) {
    $displaySkills = array_merge($displaySkills, $r['skills']);
}
$displaySkills = array_unique($displaySkills);
sort($displaySkills);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PDF Skills Extractor</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&family=Space+Grotesk:wght@500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="style.css">
    <script>
        // Show loader immediately when page starts loading
        document.addEventListener('DOMContentLoaded', function() {
            const loader = document.createElement('div');
            loader.id = 'initialLoader';
            loader.className = 'loader-overlay active';
            loader.innerHTML = `
                <div class="loader-container">
                    <div class="loader-spinner"></div>
                    <div class="loader-text">Loading PDF Skills...</div>
                    <div class="progress-bar-container">
                        <div class="progress-bar"></div>
                    </div>
                    <div class="progress-text">Processing PDFs...</div>
                </div>
            `;
            document.body.insertBefore(loader, document.body.firstChild);
            
            // Hide loader after page is fully loaded
            window.addEventListener('load', function() {
                setTimeout(function() {
                    loader.classList.remove('active');
                    setTimeout(function() {
                        if (loader.parentNode) {
                            loader.parentNode.removeChild(loader);
                        }
                    }, 300);
                }, 500);
            });
        });
    </script>
    <style>
        :root {
            --color-dark: #020617;
            --color-mid: #0f172a;
            --color-white: #f8fafc;
            --royal-blue: #3b82f6;
            --royal-blue-light: #60a5fa;
            --color-accent: #38bdf8;
            --purple: #8b5cf6;
            --color-muted: rgba(248, 250, 252, 0.65);
            --gradient-royal: linear-gradient(135deg, #1e40af 0%, #7c3aed 100%);
            --font-primary: 'Inter', system-ui, sans-serif;
            --font-heading: 'Space Grotesk', sans-serif;
        }
        
        body.resume-index {
            font-family: var(--font-primary);
            background: var(--color-dark);
            color: var(--color-white);
            min-height: 100vh;
            padding: 20px;
            position: relative;
            overflow-x: hidden;
        }
        
        body.resume-index::before {
            content: '';
            position: fixed;
            inset: 0;
            z-index: -1;
            background: radial-gradient(circle at 15% 50%, rgba(29, 78, 216, 0.08) 0%, transparent 40%),
                radial-gradient(circle at 85% 30%, rgba(124, 58, 237, 0.08) 0%, transparent 40%);
            pointer-events: none;
        }
        
        .skills-container {
            width: 90%;
            max-width: none;
            margin: 20px auto;
            padding: 20px;
        }
        
        .page-header {
            background: rgba(15, 23, 42, 0.6);
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px);
            border: 1px solid rgba(255, 255, 255, 0.08);
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 30px;
            text-align: center;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.2);
        }
        
        .page-header h1 {
            margin-bottom: 10px;
            color: var(--color-white);
            font-family: var(--font-heading);
            font-size: 1.75rem;
        }
        
        .page-header p {
            color: var(--color-muted);
            font-size: 16px;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: rgba(15, 23, 42, 0.6);
            backdrop-filter: blur(12px);
            border: 1px solid rgba(255, 255, 255, 0.08);
            border-radius: 16px;
            padding: 25px;
            text-align: center;
            transition: all 0.3s ease;
        }
        
        .stat-card:hover {
            border-color: rgba(96, 165, 250, 0.3);
            transform: translateY(-2px);
        }
        
        .stat-number {
            font-size: 36px;
            font-weight: bold;
            background: linear-gradient(135deg, var(--royal-blue-light) 0%, var(--purple) 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .stat-label {
            color: var(--color-muted);
            font-size: 14px;
            margin-top: 8px;
        }
        
        .all-skills-section {
            background: rgba(15, 23, 42, 0.6);
            backdrop-filter: blur(12px);
            border: 1px solid rgba(255, 255, 255, 0.08);
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }
        
        .all-skills-section h2 {
            color: var(--color-white);
            margin-bottom: 20px;
            font-size: 24px;
            font-family: var(--font-heading);
        }
        
        .all-skills-section p {
            color: var(--color-muted) !important;
        }
        
        .skills-cloud {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
        }
        
        .skill-checkbox-label {
            display: inline-flex;
            align-items: center;
            gap: 6px;
            cursor: pointer;
            margin: 0;
            user-select: none;
        }
        
        .skill-checkbox-label input {
            display: none;
        }
        
        .skill-checkbox-tag {
            background: rgba(59, 130, 246, 0.15);
            color: var(--royal-blue-light);
            padding: 4px 10px;
            border-radius: 8px;
            font-size: 11px;
            font-weight: 500;
            transition: all 0.2s ease;
            border: 1px solid rgba(96, 165, 250, 0.3);
        }
        
        .skill-checkbox-label:hover .skill-checkbox-tag {
            background: rgba(59, 130, 246, 0.25);
            border-color: rgba(96, 165, 250, 0.5);
        }
        
        .skill-checkbox-label input:checked + .skill-checkbox-tag {
            background: var(--gradient-royal);
            color: white;
            border-color: transparent;
        }
        
        .skill-filter-actions {
            display: flex;
            gap: 10px;
            margin-top: 12px;
            flex-wrap: wrap;
        }
        
        .skill-filter-clear {
            font-size: 12px;
            color: var(--royal-blue-light);
            background: none;
            border: none;
            cursor: pointer;
            text-decoration: underline;
        }
        
        .skill-filter-clear:hover {
            color: var(--color-accent);
        }
        
        .pdf-results {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 14px;
        }
        
        .pdf-results > h2,
        .pdf-results > .letter-filter,
        .pdf-results > .pagination,
        .pdf-results > p {
            grid-column: 1 / -1;
        }
        
        .pdf-results > h2 {
            color: var(--color-white) !important;
        }
        
        .pdf-card {
            background: rgba(15, 23, 42, 0.6);
            backdrop-filter: blur(12px);
            border: 1px solid rgba(255, 255, 255, 0.08);
            border-radius: 14px;
            padding: 14px 16px;
            transition: all 0.3s ease;
        }
        
        .pdf-card:hover {
            transform: translateY(-4px);
            border-color: rgba(96, 165, 250, 0.3);
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.2), 0 0 15px rgba(59, 130, 246, 0.15);
        }
        
        .pdf-header {
            display: flex;
            flex-direction: column;
            gap: 8px;
            margin-bottom: 10px;
        }
        
        .pdf-name {
            font-size: 15px;
            font-weight: 600;
            color: var(--color-white);
            word-break: break-word;
            line-height: 1.3;
            width: 100%;
        }
        
        .pdf-info {
            display: flex;
            gap: 10px;
            font-size: 11px;
            color: var(--color-muted);
            flex-wrap: wrap;
        }
        
        .info-badge {
            background: rgba(255, 255, 255, 0.08);
            padding: 3px 8px;
            border-radius: 6px;
            font-size: 11px;
            color: var(--color-muted);
        }
        
        .skills-list {
            display: flex;
            flex-wrap: wrap;
            gap: 4px;
            margin-top: 8px;
            line-height: 1.4;
        }
        
        .skill-badge {
            background: rgba(59, 130, 246, 0.2);
            color: var(--royal-blue-light);
            padding: 2px 6px;
            border-radius: 6px;
            font-size: 10px;
            font-weight: 500;
            display: inline-block;
            border: 1px solid rgba(96, 165, 250, 0.25);
        }
        
        .skill-badge.skill-more {
            background: rgba(255, 255, 255, 0.08);
            color: var(--color-muted);
            font-style: italic;
            cursor: pointer;
            border: 1px solid rgba(255, 255, 255, 0.1);
            font-family: inherit;
            transition: all 0.2s;
        }
        
        .skill-badge.skill-more:hover {
            background: var(--royal-blue);
            color: white;
            border-color: transparent;
        }
        
        .skills-lightbox {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(2, 6, 23, 0.85);
            backdrop-filter: blur(8px);
            z-index: 9999;
            align-items: center;
            justify-content: center;
            padding: 20px;
            box-sizing: border-box;
        }
        
        .skills-lightbox.active {
            display: flex;
        }
        
        .skills-lightbox-content {
            background: var(--color-mid);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 16px;
            padding: 24px 28px;
            max-width: 500px;
            width: 100%;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 25px 50px rgba(0, 0, 0, 0.4);
            position: relative;
        }
        
        .skills-lightbox-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 16px;
            padding-bottom: 12px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .skills-lightbox-title {
            font-size: 18px;
            font-weight: 600;
            color: var(--color-white);
        }
        
        .skills-lightbox-close {
            background: rgba(255, 255, 255, 0.08);
            border: 1px solid rgba(255, 255, 255, 0.1);
            width: 32px;
            height: 32px;
            border-radius: 50%;
            font-size: 18px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--color-muted);
            transition: all 0.2s;
        }
        
        .skills-lightbox-close:hover {
            background: var(--royal-blue);
            color: white;
            border-color: transparent;
        }
        
        .skills-lightbox-list {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
        }
        
        .skills-lightbox-list .skill-badge {
            font-size: 12px;
            padding: 4px 10px;
        }
        
        .no-skills {
            color: var(--color-muted);
            font-style: italic;
            padding: 8px;
            font-size: 11px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 8px;
        }
        
        .error-message, .warning-message {
            background: rgba(30, 58, 138, 0.3);
            border: 1px solid rgba(99, 102, 241, 0.4);
            color: var(--royal-blue-light);
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .error-message {
            background: rgba(185, 28, 28, 0.2);
            border-color: rgba(239, 68, 68, 0.5);
            color: #fca5a5;
        }
        
        .letter-filter {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
            margin-bottom: 25px;
            justify-content: center;
        }
        
        .letter-btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 36px;
            height: 36px;
            min-width: 36px;
            padding: 0;
            border: 1px solid rgba(255, 255, 255, 0.15);
            background: rgba(255, 255, 255, 0.05);
            color: var(--color-muted);
            font-size: 14px;
            font-weight: 600;
            text-decoration: none;
            border-radius: 8px;
            transition: all 0.2s ease;
        }
        
        .letter-btn:hover {
            border-color: var(--royal-blue);
            color: var(--royal-blue-light);
            background: rgba(59, 130, 246, 0.15);
        }
        
        .letter-btn.active {
            background: var(--gradient-royal);
            border-color: transparent;
            color: white;
        }
        
        .letter-btn:not(.has-resumes) {
            opacity: 0.4;
            cursor: not-allowed;
            pointer-events: none;
        }
        
        .pagination {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            margin-top: 25px;
            flex-wrap: wrap;
        }
        
        .pagination a, .pagination span {
            padding: 10px 16px;
            border: 1px solid rgba(255, 255, 255, 0.15);
            background: rgba(255, 255, 255, 0.05);
            color: var(--color-muted);
            text-decoration: none;
            border-radius: 8px;
            font-weight: 500;
            transition: all 0.2s ease;
        }
        
        .pagination a:hover {
            border-color: var(--royal-blue);
            color: var(--royal-blue-light);
            background: rgba(59, 130, 246, 0.15);
        }
        
        .pagination span.current {
            background: var(--gradient-royal);
            border-color: transparent;
            color: white;
        }
        
        .pagination span.ellipsis {
            border: none;
            background: transparent;
            color: var(--color-muted);
        }
        
        .install-command {
            background: var(--color-mid);
            color: var(--royal-blue-light);
            padding: 12px 20px;
            border-radius: 8px;
            margin: 15px 0;
            font-family: 'Courier New', monospace;
            font-size: 14px;
            overflow-x: auto;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .back-link {
            display: inline-block;
            margin-top: 20px;
            padding: 12px 30px;
            background: var(--gradient-royal);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            transition: all 0.2s ease;
            font-weight: 600;
            border: none;
        }
        
        .back-link:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(59, 130, 246, 0.4);
        }
        
        body.resume-index .skills-container p[style*="color: #888"] {
            color: var(--color-muted) !important;
        }
        
        @media (max-width: 1024px) {
            .pdf-results {
                grid-template-columns: repeat(2, 1fr);
            }
        }
        
        @media (max-width: 768px) {
            .skills-container {
                padding: 10px;
                width: 95%;
            }
            
            .pdf-results {
                grid-template-columns: 1fr;
            }
            
            .pdf-header {
                align-items: flex-start;
            }
        }
    </style>
</head>
<body class="resume-index">
    <div class="skills-container">
        <div class="page-header">
            <h1>📄 PDF Skills Extractor</h1>
            <p>Automatically extract and display skills from resume PDFs</p>
        </div>
        
        <?php if (!empty($skippedFiles)): ?>
            <div class="warning-message">
                <h3>⚠️ Some Files Were Skipped</h3>
                <p>The following PDFs were too large to process (limit: 10MB):</p>
                <ul style="list-style: none; padding: 0; margin-top: 10px;">
                    <?php foreach ($skippedFiles as $skipped): ?>
                        <li style="margin: 5px 0;">
                            <strong><?php echo htmlspecialchars($skipped['name']); ?></strong> 
                            - <?php echo $skipped['size']; ?>
                        </li>
                    <?php endforeach; ?>
                </ul>
                <p style="margin-top: 15px; font-size: 13px;">
                    💡 Tip: Try compressing these PDFs or contact your administrator to increase the memory limit.
                </p>
            </div>
        <?php endif; ?>
        
        <?php if (!$parserAvailable): ?>
            <div class="error-message">
                <h3>⚠️ PDF Parser Not Installed</h3>
                <p>To use this feature, you need to install the PDF parser library.</p>
                
                <h4 style="margin-top: 20px;">📦 Option 1: Using Composer (Recommended)</h4>
                <p>Run the following command in your project directory:</p>
                <div class="install-command">composer install</div>
                <p style="text-align: center; margin-top: 20px;">
                    <a href="install_composer_guide.html" style="display: inline-block; padding: 12px 25px; background: #4CAF50; color: white; text-decoration: none; border-radius: 8px; font-weight: 600; margin: 5px;">
                        📖 Composer Installation Guide
                    </a>
                </p>
                
                <h4 style="margin-top: 30px;">🔧 Option 2: Manual Installation</h4>
                <p>If you cannot install Composer, try the manual installation method:</p>
                <p style="text-align: center; margin-top: 15px;">
                    <a href="manual_install.php" style="display: inline-block; padding: 12px 25px; background: #2196F3; color: white; text-decoration: none; border-radius: 8px; font-weight: 600; margin: 5px;">
                        🛠️ Manual Installation Tool
                    </a>
                </p>
            </div>
        <?php elseif (empty($pdfFiles)): ?>
            <div class="warning-message">
                <h3>📁 No PDF Files Found</h3>
                <p>Please add PDF files to the <strong>pdfs</strong> folder to extract skills.</p>
            </div>
        <?php else: ?>
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-number"><?php echo count($pdfFiles); ?></div>
                    <div class="stat-label">Total PDFs</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo count($allSkills); ?></div>
                    <div class="stat-label">Unique Skills</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php 
                        $totalSkills = array_sum(array_column($results, 'skillCount'));
                        echo $totalSkills;
                    ?></div>
                    <div class="stat-label">Total Skills Found</div>
                </div>
            </div>
            
            <?php if (!empty($displaySkills)): ?>
            <div class="all-skills-section">
                <h2>🎯 Skills Available in Resumes</h2>
                <p style="color: #666; font-size: 13px; margin-bottom: 12px;">Select skills to filter resumes below (resume must have all selected skills)</p>
                <form method="get" id="skillFilterForm" class="skill-filter-form">
                    <?php if ($letterFilter): ?><input type="hidden" name="letter" value="<?php echo htmlspecialchars($letterFilter); ?>"><?php endif; ?>
                    <div class="skills-cloud">
                        <?php foreach ($displaySkills as $skill): 
                            $isChecked = in_array($skill, $selectedSkills);
                        ?>
                            <label class="skill-checkbox-label">
                                <input type="checkbox" name="skills[]" value="<?php echo htmlspecialchars($skill); ?>" <?php echo $isChecked ? 'checked' : ''; ?> onchange="this.form.submit()">
                                <span class="skill-checkbox-tag"><?php echo htmlspecialchars($skill); ?></span>
                            </label>
                        <?php endforeach; ?>
                    </div>
                    <?php if (!empty($selectedSkills)): ?>
                    <div class="skill-filter-actions">
                        <a href="index.php<?php echo $letterFilter ? '?letter=' . urlencode($letterFilter) : ''; ?>" class="skill-filter-clear">Clear skill filter</a>
                    </div>
                    <?php endif; ?>
                </form>
            </div>
            <?php endif; ?>
            
            <?php
            // Count resumes per letter (considering skill filter)
            $letterCounts = [];
            foreach ($results as $r) {
                if (!empty($selectedSkills)) {
                    $cardSkills = array_map('strtolower', $r['skills']);
                    $hasAll = true;
                    foreach ($selectedSkills as $sk) {
                        if (!in_array(strtolower($sk), $cardSkills)) { $hasAll = false; break; }
                    }
                    if (!$hasAll) continue;
                }
                $displayName = $r['displayName'] ?? pathinfo($r['name'], PATHINFO_FILENAME);
                $firstChar = strtoupper(substr(trim($displayName), 0, 1));
                if (ctype_alpha($firstChar)) {
                    $letterCounts[$firstChar] = ($letterCounts[$firstChar] ?? 0) + 1;
                } else {
                    $letterCounts['#'] = ($letterCounts['#'] ?? 0) + 1;
                }
            }
            $urlSkillSuffix = $skillParams ? '&' . $skillParams : '';
            ?>
            
            <div class="pdf-results">
                <h2 style="color: #333; margin-bottom: 15px;">📋 Resumes (<?php echo $totalResumes; ?> total<?php echo $letterFilter ? " - Letter $letterFilter" : ''; ?><?php echo !empty($selectedSkills) ? ' - ' . count($selectedSkills) . ' skill(s) filter' : ''; ?>)</h2>
                
                <div class="letter-filter">
                    <a href="index.php?page=1<?php echo $urlSkillSuffix ? '&' . $urlSkillSuffix : ''; ?>" class="letter-btn <?php echo !$letterFilter ? 'active' : ''; ?>" title="All resumes">All</a>
                    <?php for ($i = ord('A'); $i <= ord('Z'); $i++): 
                        $letter = chr($i);
                        $count = $letterCounts[$letter] ?? 0;
                        $hasResumes = $count > 0;
                        $isActive = $letterFilter === $letter;
                    ?>
                        <a href="index.php?letter=<?php echo $letter; ?>&page=1<?php echo $urlSkillSuffix ? '&' . $urlSkillSuffix : ''; ?>" 
                           class="letter-btn <?php echo $isActive ? 'active' : ''; ?> <?php echo $hasResumes ? 'has-resumes' : ''; ?>"
                           title="<?php echo $hasResumes ? "$count resume(s)" : 'No resumes'; ?>">
                            <?php echo $letter; ?>
                        </a>
                    <?php endfor; ?>
                </div>
                
                <?php if (empty($paginatedResults)): ?>
                    <div class="no-skills" style="grid-column: 1 / -1;">
                        <?php 
                        $msg = $letterFilter ? "No resumes found starting with '$letterFilter'" : 'No resumes found';
                        if (!empty($selectedSkills)) {
                            $msg .= " with selected skills: " . implode(', ', $selectedSkills);
                        }
                        echo $msg . '.';
                        ?>
                    </div>
                <?php else: ?>
                <?php foreach ($paginatedResults as $result): ?>
                    <div class="pdf-card">
                        <div class="pdf-header">
                            <div class="pdf-name" title="<?php echo htmlspecialchars($result['name']); ?>">
                                <?php echo htmlspecialchars($result['displayName'] ?? $result['name']); ?>
                            </div>
                            <div class="pdf-info">
                                <span class="info-badge"><?php echo $result['size']; ?></span>
                                <span class="info-badge"><?php echo $result['skillCount']; ?> skills</span>
                            </div>
                        </div>
                        
                        <?php if ($result['success'] && !empty($result['skills'])): 
                            $maxSkills = 6;
                            $skills = $result['skills'];
                            $displaySkills = array_slice($skills, 0, $maxSkills);
                            $moreCount = count($skills) - $maxSkills;
                        ?>
                            <div class="skills-list">
                                <?php foreach ($displaySkills as $skill): ?>
                                    <span class="skill-badge"><?php echo htmlspecialchars($skill); ?></span>
                                <?php endforeach; ?>
                                <?php if ($moreCount > 0): ?>
                                    <button type="button" class="skill-badge skill-more" 
                                            data-name="<?php echo htmlspecialchars($result['displayName'] ?? $result['name']); ?>"
                                            data-skills="<?php echo htmlspecialchars(json_encode($skills)); ?>"
                                            title="View all skills">+<?php echo $moreCount; ?></button>
                                <?php endif; ?>
                            </div>
                        <?php elseif ($result['success'] && empty($result['skills'])): ?>
                            <div class="no-skills">No skills found in this PDF</div>
                        <?php else: ?>
                            <div class="no-skills">Error reading PDF file</div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
                <?php endif; ?>
                
                <?php if ($totalPages > 1): 
                    $baseUrl = 'index.php?' . ($letterFilter ? 'letter=' . $letterFilter . '&' : '') . $skillParams . ($skillParams ? '&' : '');
                ?>
                <div class="pagination">
                    <?php if ($currentPage > 1): ?>
                        <a href="<?php echo $baseUrl; ?>page=<?php echo $currentPage - 1; ?>">← Prev</a>
                    <?php endif; ?>
                    
                    <?php
                    $startPage = max(1, $currentPage - 2);
                    $endPage = min($totalPages, $currentPage + 2);
                    if ($startPage > 1): ?>
                        <a href="<?php echo $baseUrl; ?>page=1">1</a>
                        <?php if ($startPage > 2): ?><span class="ellipsis">...</span><?php endif; ?>
                    <?php endif;
                    
                    for ($p = $startPage; $p <= $endPage; $p++): ?>
                        <?php if ($p == $currentPage): ?>
                            <span class="current"><?php echo $p; ?></span>
                        <?php else: ?>
                            <a href="<?php echo $baseUrl; ?>page=<?php echo $p; ?>"><?php echo $p; ?></a>
                        <?php endif; ?>
                    <?php endfor;
                    
                    if ($endPage < $totalPages): ?>
                        <?php if ($endPage < $totalPages - 1): ?><span class="ellipsis">...</span><?php endif; ?>
                        <a href="<?php echo $baseUrl; ?>page=<?php echo $totalPages; ?>"><?php echo $totalPages; ?></a>
                    <?php endif; ?>
                    
                    <?php if ($currentPage < $totalPages): ?>
                        <a href="<?php echo $baseUrl; ?>page=<?php echo $currentPage + 1; ?>">Next →</a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                <?php if ($totalResumes > 0): ?>
                <p style="text-align: center; color: #888; margin-top: 10px; font-size: 14px;">
                    Page <?php echo $currentPage; ?> of <?php echo $totalPages; ?> 
                    (Showing <?php echo count($paginatedResults); ?> of <?php echo $totalResumes; ?> resumes)
                </p>
                <?php endif; ?>
            </div>
        <?php endif; ?>
        
        <div style="text-align: center;">
            <a href="search_skills.php" class="back-link" style="margin-right: 10px;">🔍 Search Skills</a>
            <a href="index.php" class="back-link">← Back to Home</a>
        </div>
    </div>
    
    <!-- Skills Lightbox -->
    <div class="skills-lightbox" id="skillsLightbox" role="dialog" aria-label="All skills">
        <div class="skills-lightbox-content" onclick="event.stopPropagation()">
            <div class="skills-lightbox-header">
                <h3 class="skills-lightbox-title" id="skillsLightboxTitle">All Skills</h3>
                <button type="button" class="skills-lightbox-close" id="skillsLightboxClose" aria-label="Close">&times;</button>
            </div>
            <div class="skills-lightbox-list" id="skillsLightboxList"></div>
        </div>
    </div>
    
    <!-- Loader Overlay -->
    <div class="loader-overlay" id="loaderOverlay">
        <div class="loader-container">
            <div class="loader-spinner"></div>
            <div class="loader-text" id="loaderText">Loading...</div>
            <div class="progress-bar-container">
                <div class="progress-bar"></div>
            </div>
            <div class="progress-text">Please wait...</div>
        </div>
    </div>
    
    <script>
        // Show loader on page load
        window.addEventListener('load', function() {
            const loader = document.getElementById('loaderOverlay');
            if (loader) {
                loader.classList.remove('active');
            }
        });
        
        // Show loader on navigation links
        document.addEventListener('DOMContentLoaded', function() {
            const links = document.querySelectorAll('a[href$=".php"]');
            links.forEach(link => {
                link.addEventListener('click', function(e) {
                    const loader = document.getElementById('loaderOverlay');
                    const loaderText = document.getElementById('loaderText');
                    loaderText.textContent = 'Loading page...';
                    loader.classList.add('active');
                });
            });
            
            // Skills lightbox
            const lightbox = document.getElementById('skillsLightbox');
            const lightboxTitle = document.getElementById('skillsLightboxTitle');
            const lightboxList = document.getElementById('skillsLightboxList');
            const lightboxClose = document.getElementById('skillsLightboxClose');
            
            function openSkillsLightbox(name, skills) {
                lightboxTitle.textContent = name + ' – All Skills';
                lightboxList.innerHTML = '';
                skills.forEach(function(skill) {
                    const badge = document.createElement('span');
                    badge.className = 'skill-badge';
                    badge.textContent = skill;
                    lightboxList.appendChild(badge);
                });
                lightbox.classList.add('active');
                document.body.style.overflow = 'hidden';
            }
            
            function closeSkillsLightbox() {
                lightbox.classList.remove('active');
                document.body.style.overflow = '';
            }
            
            document.querySelectorAll('.skill-more').forEach(function(btn) {
                btn.addEventListener('click', function(e) {
                    e.preventDefault();
                    const name = this.getAttribute('data-name');
                    const skills = JSON.parse(this.getAttribute('data-skills') || '[]');
                    openSkillsLightbox(name, skills);
                });
            });
            
            if (lightboxClose) lightboxClose.addEventListener('click', closeSkillsLightbox);
            lightbox.addEventListener('click', function(e) {
                if (e.target === lightbox) closeSkillsLightbox();
            });
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape' && lightbox.classList.contains('active')) closeSkillsLightbox();
            });
        });
    </script>
</body>
</html>

