<?php
/**
 * Manual Installation Script for PDF Parser
 * This script will download and install the PDF parser library without Composer
 */

// Set error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Check if vendor directory already exists
$vendorExists = file_exists('vendor/autoload.php');

// Process installation
$message = '';
$messageType = '';

if (isset($_POST['install'])) {
    $message = installPDFParser();
    $messageType = strpos($message, 'Success') !== false ? 'success' : 'error';
    $vendorExists = file_exists('vendor/autoload.php');
}

function installPDFParser() {
    // Create vendor directory structure
    $vendorDir = 'vendor/smalot/pdfparser';
    
    if (!file_exists('vendor')) {
        if (!mkdir('vendor', 0755, true)) {
            return 'Error: Failed to create vendor directory. Check permissions.';
        }
    }
    
    // Download the library as ZIP
    $zipUrl = 'https://github.com/smalot/pdfparser/archive/refs/tags/v2.7.0.zip';
    $zipFile = 'vendor/pdfparser.zip';
    
    // Download using file_get_contents (works if allow_url_fopen is enabled)
    if (ini_get('allow_url_fopen')) {
        $zipContent = @file_get_contents($zipUrl);
        if ($zipContent === false) {
            return 'Error: Failed to download PDF parser library. Check your internet connection.';
        }
        
        if (!file_put_contents($zipFile, $zipContent)) {
            return 'Error: Failed to save downloaded file. Check write permissions.';
        }
        
        // Extract ZIP file
        $zip = new ZipArchive();
        if ($zip->open($zipFile) === true) {
            $extractPath = 'vendor/';
            $zip->extractTo($extractPath);
            $zip->close();
            
            // Move extracted files to correct location
            $extractedDir = 'vendor/pdfparser-2.7.0';
            if (file_exists($extractedDir)) {
                if (!file_exists(dirname($vendorDir))) {
                    mkdir(dirname($vendorDir), 0755, true);
                }
                rename($extractedDir, $vendorDir);
            }
            
            // Create a simple autoloader
            createAutoloader();
            
            // Clean up
            @unlink($zipFile);
            
            return 'Success! PDF Parser library installed successfully. You can now use the Skills Extractor!';
        } else {
            return 'Error: Failed to extract ZIP file. Make sure ZIP extension is enabled in PHP.';
        }
    } else {
        return 'Error: allow_url_fopen is disabled. Please use Composer instead or enable allow_url_fopen in php.ini.';
    }
}

function createAutoloader() {
    $autoloadContent = '<?php
// Simple autoloader for PDF Parser
spl_autoload_register(function ($class) {
    $prefix = "Smalot\\\\PdfParser\\\\";
    $base_dir = __DIR__ . "/smalot/pdfparser/src/Smalot/PdfParser/";
    
    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }
    
    $relative_class = substr($class, $len);
    $file = $base_dir . str_replace("\\\\", "/", $relative_class) . ".php";
    
    if (file_exists($file)) {
        require $file;
    }
});
';
    
    file_put_contents('vendor/autoload.php', $autoloadContent);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manual Installation - PDF Parser</title>
    <link rel="stylesheet" href="style.css">
    <style>
        .install-container {
            max-width: 800px;
            margin: 40px auto;
            padding: 20px;
        }
        
        .install-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
            padding: 40px;
        }
        
        .install-card h1 {
            color: #333;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .install-card h2 {
            color: #667eea;
            margin-top: 30px;
            margin-bottom: 15px;
        }
        
        .status-box {
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
            text-align: center;
        }
        
        .status-success {
            background: #d4edda;
            border: 2px solid #28a745;
            color: #155724;
        }
        
        .status-error {
            background: #f8d7da;
            border: 2px solid #f44336;
            color: #721c24;
        }
        
        .status-warning {
            background: #fff3cd;
            border: 2px solid #ffc107;
            color: #856404;
        }
        
        .status-installed {
            background: #d1ecf1;
            border: 2px solid #17a2b8;
            color: #0c5460;
        }
        
        .install-btn {
            display: block;
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 18px;
            font-weight: 600;
            cursor: pointer;
            margin: 20px 0;
            transition: transform 0.2s ease;
        }
        
        .install-btn:hover {
            transform: translateY(-2px);
        }
        
        .install-btn:disabled {
            background: #ccc;
            cursor: not-allowed;
        }
        
        .link-btn {
            display: inline-block;
            padding: 12px 25px;
            background: #667eea;
            color: white;
            text-decoration: none;
            border-radius: 8px;
            margin: 10px 5px;
            font-weight: 600;
        }
        
        .requirements {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
        }
        
        .requirements h3 {
            color: #555;
            margin-bottom: 10px;
        }
        
        .requirements ul {
            list-style: none;
            padding: 0;
        }
        
        .requirements li {
            padding: 8px 0;
            color: #666;
        }
        
        .check-ok {
            color: #28a745;
            font-weight: bold;
        }
        
        .check-fail {
            color: #f44336;
            font-weight: bold;
        }
    </style>
</head>
<body>
    <div class="install-container">
        <div class="install-card">
            <h1>📦 Manual PDF Parser Installation</h1>
            
            <?php if ($message): ?>
                <div class="status-box status-<?php echo $messageType; ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($vendorExists): ?>
                <div class="status-box status-installed">
                    <h3>✅ PDF Parser Already Installed!</h3>
                    <p>The PDF parser library is already installed and ready to use.</p>
                </div>
                
                <div style="text-align: center;">
                    <a href="view_skills.php" class="link-btn">🎯 Go to Skills Extractor</a>
                    <a href="index.php" class="link-btn">🏠 Back to Home</a>
                </div>
            <?php else: ?>
                <div class="status-box status-warning">
                    <h3>⚠️ PDF Parser Not Installed</h3>
                    <p>This installation method is a fallback option. Using Composer is strongly recommended.</p>
                </div>
                
                <h2>📋 System Requirements Check</h2>
                <div class="requirements">
                    <ul>
                        <li>
                            PHP Version: <strong><?php echo PHP_VERSION; ?></strong>
                            <?php echo version_compare(PHP_VERSION, '7.4.0', '>=') ? '<span class="check-ok">✓ OK</span>' : '<span class="check-fail">✗ Need 7.4+</span>'; ?>
                        </li>
                        <li>
                            allow_url_fopen: <strong><?php echo ini_get('allow_url_fopen') ? 'Enabled' : 'Disabled'; ?></strong>
                            <?php echo ini_get('allow_url_fopen') ? '<span class="check-ok">✓ OK</span>' : '<span class="check-fail">✗ Required</span>'; ?>
                        </li>
                        <li>
                            ZIP Extension: <strong><?php echo extension_loaded('zip') ? 'Loaded' : 'Not Loaded'; ?></strong>
                            <?php echo extension_loaded('zip') ? '<span class="check-ok">✓ OK</span>' : '<span class="check-fail">✗ Required</span>'; ?>
                        </li>
                        <li>
                            Write Permission: <strong><?php echo is_writable(__DIR__) ? 'Yes' : 'No'; ?></strong>
                            <?php echo is_writable(__DIR__) ? '<span class="check-ok">✓ OK</span>' : '<span class="check-fail">✗ Required</span>'; ?>
                        </li>
                    </ul>
                </div>
                
                <?php 
                $canInstall = version_compare(PHP_VERSION, '7.4.0', '>=') 
                    && ini_get('allow_url_fopen') 
                    && extension_loaded('zip') 
                    && is_writable(__DIR__);
                ?>
                
                <?php if ($canInstall): ?>
                    <h2>🚀 Install PDF Parser</h2>
                    <p>Click the button below to download and install the PDF parser library:</p>
                    
                    <form method="post">
                        <button type="submit" name="install" class="install-btn">
                            📥 Download and Install PDF Parser
                        </button>
                    </form>
                    
                    <p style="text-align: center; color: #888; font-size: 14px;">
                        This will download the library from GitHub and install it automatically.
                    </p>
                <?php else: ?>
                    <div class="status-box status-error">
                        <h3>❌ Cannot Install Automatically</h3>
                        <p>Some system requirements are not met. Please install Composer instead.</p>
                    </div>
                <?php endif; ?>
                
                <h2>💡 Recommended: Use Composer</h2>
                <p>For better reliability and easier updates, we recommend installing Composer:</p>
                
                <div style="text-align: center;">
                    <a href="install_composer_guide.html" class="link-btn">📖 View Composer Installation Guide</a>
                    <a href="https://getcomposer.org/download/" class="link-btn" target="_blank">⬇️ Download Composer</a>
                </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>

