<?php
require_once 'config.php';

// Check if PDF parser library is available
$parserAvailable = file_exists('vendor/autoload.php');

if ($parserAvailable) {
    require_once 'vendor/autoload.php';
}

// Increase execution time for large PDF processing
set_time_limit(300); // 5 minutes max
ini_set('max_execution_time', 300);

// Handle cache clearing
if (isset($_GET['clear_cache']) && $_GET['clear_cache'] == '1') {
    $cacheFile = 'pdfs/skills_cache.json';
    if (file_exists($cacheFile)) {
        unlink($cacheFile);
    }
    // Redirect to remove the query parameter
    header('Location: search_skills.php');
    exit;
}

/**
 * Extract text from PDF using smalot/pdfparser
 */
function extractTextFromPDF($pdfPath) {
    try {
        // Check file size - skip files larger than 10MB
        $fileSize = filesize($pdfPath);
        if ($fileSize > 10 * 1024 * 1024) {
            return null; // Skip very large files
        }
        
        // Increase memory limit temporarily for PDF processing
        $oldMemoryLimit = ini_get('memory_limit');
        ini_set('memory_limit', '256M');
        
        // Suppress deprecation warnings from PDF parser
        $oldErrorReporting = error_reporting();
        error_reporting(E_ERROR | E_PARSE);
        
        $parser = new \Smalot\PdfParser\Parser();
        $pdf = $parser->parseFile($pdfPath);
        $text = $pdf->getText();
        
        // Restore settings
        error_reporting($oldErrorReporting);
        ini_set('memory_limit', $oldMemoryLimit);
        
        return $text;
    } catch (Exception $e) {
        // Restore settings in case of exception
        if (isset($oldErrorReporting)) {
            error_reporting($oldErrorReporting);
        }
        if (isset($oldMemoryLimit)) {
            ini_set('memory_limit', $oldMemoryLimit);
        }
        return null;
    } catch (Error $e) {
        // Catch fatal errors like memory exhaustion
        if (isset($oldErrorReporting)) {
            error_reporting($oldErrorReporting);
        }
        if (isset($oldMemoryLimit)) {
            ini_set('memory_limit', $oldMemoryLimit);
        }
        return null;
    }
}

/**
 * Common skills to look for in resumes
 */
function getCommonSkills() {
    return [
        // Programming Languages
        'PHP', 'JavaScript', 'Python', 'Java', 'C++', 'C#', 'Ruby', 'Go', 'Swift', 'Kotlin',
        'TypeScript', 'Rust', 'Scala', 'Perl', 'R', 'Dart', 'Objective-C',
        
        // Web Technologies
        'HTML', 'HTML5', 'CSS', 'CSS3', 'SASS', 'SCSS', 'LESS', 'Bootstrap', 'Tailwind',
        'React', 'Angular', 'Vue.js', 'Vue', 'Next.js', 'Nuxt.js', 'jQuery', 'AJAX',
        'Node.js', 'Express.js', 'Laravel', 'Django', 'Flask', 'Spring', 'ASP.NET',
        
        // Databases
        'MySQL', 'PostgreSQL', 'MongoDB', 'Redis', 'SQLite', 'Oracle', 'SQL Server',
        'MariaDB', 'Cassandra', 'DynamoDB', 'Firebase', 'Elasticsearch',
        
        // DevOps & Tools
        'Git', 'GitHub', 'GitLab', 'Docker', 'Kubernetes', 'Jenkins', 'CI/CD',
        'AWS', 'Azure', 'Google Cloud', 'GCP', 'Linux', 'Unix', 'Bash',
        'Terraform', 'Ansible', 'Nginx', 'Apache',
        
        // Mobile Development
        'Android', 'iOS', 'React Native', 'Flutter', 'Xamarin', 'Ionic',
        
        // Testing & Quality
        'Jest', 'Mocha', 'PHPUnit', 'Selenium', 'JUnit', 'Cypress', 'Testing',
        'Unit Testing', 'Integration Testing', 'TDD', 'BDD',
        
        // Design & UI/UX
        'Photoshop', 'Illustrator', 'Figma', 'Sketch', 'Adobe XD', 'UI/UX',
        'Responsive Design', 'Web Design', 'Graphic Design',
        
        // Other Technologies
        'REST API', 'RESTful', 'GraphQL', 'Microservices', 'API', 'JSON', 'XML',
        'Agile', 'Scrum', 'Jira', 'Webpack', 'Babel', 'NPM', 'Composer',
        'WordPress', 'Shopify', 'Magento', 'WooCommerce',
        
        // Soft Skills
        'Leadership', 'Communication', 'Team Work', 'Problem Solving', 'Project Management',
        'Time Management', 'Critical Thinking', 'Collaboration'
    ];
}

/**
 * Extract skills from text
 */
function extractSkills($text) {
    $commonSkills = getCommonSkills();
    $foundSkills = [];
    
    // Convert text to lowercase for case-insensitive matching
    $textLower = strtolower($text);
    
    foreach ($commonSkills as $skill) {
        $skillLower = strtolower($skill);
        // Use word boundary to avoid partial matches
        if (preg_match('/\b' . preg_quote($skillLower, '/') . '\b/i', $textLower)) {
            $foundSkills[] = $skill;
        }
    }
    
    return array_unique($foundSkills);
}

/**
 * Get cache file path
 */
function getCacheFilePath() {
    return 'pdfs/skills_cache.json';
}

/**
 * Load cached PDF data
 */
function loadCache() {
    $cacheFile = getCacheFilePath();
    if (file_exists($cacheFile)) {
        $data = file_get_contents($cacheFile);
        return json_decode($data, true) ?: [];
    }
    return [];
}

/**
 * Save data to cache
 */
function saveCache($data) {
    $cacheFile = getCacheFilePath();
    file_put_contents($cacheFile, json_encode($data, JSON_PRETTY_PRINT));
}

/**
 * Get all PDFs from the pdfs folder
 */
function getPDFFiles() {
    $pdfDir = 'pdfs';
    $pdfFiles = [];
    
    if (!is_dir($pdfDir)) {
        return $pdfFiles;
    }
    
    $files = scandir($pdfDir);
    foreach ($files as $file) {
        if (pathinfo($file, PATHINFO_EXTENSION) === 'pdf') {
            $pdfFiles[] = [
                'name' => $file,
                'path' => $pdfDir . '/' . $file,
                'size' => filesize($pdfDir . '/' . $file),
                'modified' => filemtime($pdfDir . '/' . $file)
            ];
        }
    }
    
    return $pdfFiles;
}

/**
 * Format file size
 */
function formatFileSize($bytes) {
    if ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    }
    return $bytes . ' bytes';
}

// Process PDFs and extract skills with caching
$pdfFiles = getPDFFiles();
$pdfData = [];
$allSkills = [];
$skippedFiles = [];
$processedCount = 0;
$cachedCount = 0;
$maxProcessPerLoad = 50; // Limit processing to prevent timeout

if ($parserAvailable && !empty($pdfFiles)) {
    // Load cached data
    $cache = loadCache();
    $cacheUpdated = false;
    $startTime = time();
    $timeLimit = 240; // 4 minutes to be safe
    
    foreach ($pdfFiles as $pdf) {
        // Check execution time limit
        if ((time() - $startTime) > $timeLimit) {
            break;
        }
        
        // Check if file is too large (>10MB)
        if ($pdf['size'] > 10 * 1024 * 1024) {
            $skippedFiles[] = [
                'name' => $pdf['name'],
                'size' => formatFileSize($pdf['size']),
                'reason' => 'File too large (>10MB)'
            ];
            continue;
        }
        
        // Check if PDF is in cache and not modified
        $cacheKey = md5($pdf['name']);
        $useCache = false;
        
        if (isset($cache[$cacheKey]) && 
            $cache[$cacheKey]['modified'] == $pdf['modified'] &&
            $cache[$cacheKey]['size'] == $pdf['size']) {
            // Use cached data
            $pdfData[] = $cache[$cacheKey];
            $allSkills = array_merge($allSkills, $cache[$cacheKey]['skills']);
            $cachedCount++;
            $useCache = true;
        } else {
            // Limit new processing per load
            if ($processedCount >= $maxProcessPerLoad) {
                continue;
            }
            
            // Process PDF with error handling
            try {
                $text = extractTextFromPDF($pdf['path']);
                $skills = $text ? extractSkills($text) : [];
                
                $pdfInfo = [
                    'name' => $pdf['name'],
                    'path' => $pdf['path'],
                    'size' => formatFileSize($pdf['size']),
                    'sizeBytes' => $pdf['size'],
                    'skills' => $skills,
                    'skillCount' => count($skills),
                    'parsed' => $text !== null,
                    'modified' => $pdf['modified']
                ];
                
                $pdfData[] = $pdfInfo;
                $allSkills = array_merge($allSkills, $skills);
                
                // Update cache
                $cache[$cacheKey] = $pdfInfo;
                $cacheUpdated = true;
                $processedCount++;
            } catch (Exception $e) {
                // Skip problematic PDFs
                continue;
            }
        }
    }
    
    // Save updated cache
    if ($cacheUpdated) {
        saveCache($cache);
    }
}

$allSkills = array_unique($allSkills);
sort($allSkills);

// Check if there are more PDFs to process
$totalPDFs = count($pdfFiles) - count($skippedFiles);
$remainingPDFs = $totalPDFs - count($pdfData);
$showProgressNotice = ($remainingPDFs > 0 && $parserAvailable);

// Handle search
$searchResults = [];
$selectedSkills = [];
$searchPerformed = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['search'])) {
    $selectedSkills = $_POST['skills'] ?? [];
    $searchPerformed = true;
    
    if (!empty($selectedSkills)) {
        foreach ($pdfData as $pdf) {
            // Check if PDF has ANY of the selected skills
            $matchingSkills = array_intersect($selectedSkills, $pdf['skills']);
            
            if (!empty($matchingSkills)) {
                $matchPercentage = (count($matchingSkills) / count($selectedSkills)) * 100;
                $searchResults[] = [
                    'pdf' => $pdf,
                    'matchingSkills' => $matchingSkills,
                    'matchPercentage' => $matchPercentage,
                    'matchCount' => count($matchingSkills)
                ];
            }
        }
        
        // Sort by match count (descending)
        usort($searchResults, function($a, $b) {
            return $b['matchCount'] - $a['matchCount'];
        });
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Search PDFs by Skills</title>
    <link rel="stylesheet" href="style.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&family=Space+Grotesk:wght@500;600;700&family=Montserrat:wght@600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --color-dark: #020617;
            --color-mid: #0f172a;
            --color-white: #f8fafc;
            --royal-blue: #3b82f6;
            --royal-blue-dark: #1d4ed8;
            --royal-blue-light: #60a5fa;
            --purple-light: #a78bfa;
            --color-accent: #38bdf8;
            --color-muted: rgba(248, 250, 252, 0.65);
            --gradient-royal-purple: linear-gradient(135deg, #1e40af 0%, #7c3aed 100%);
            --font-primary: 'Inter', system-ui, sans-serif;
            --font-heading: 'Space Grotesk', sans-serif;
        }
        body {
            font-family: var(--font-primary);
            background: var(--color-dark);
            color: var(--color-white);
            min-height: 100vh;
            line-height: 1.6;
            overflow-x: hidden;
            position: relative;
        }
        .premium-bg {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 0;
            overflow: hidden;
            pointer-events: none;
        }
        .premium-bg .blob {
            position: absolute;
            border-radius: 50%;
            filter: blur(100px);
            opacity: 0.2;
            animation: blob-anim 25s infinite ease-in-out;
        }
        .premium-bg .blob-1 { width: 800px; height: 800px; background: #1e3a8a; top: -300px; left: -200px; }
        .premium-bg .blob-2 { width: 600px; height: 600px; background: #5b21b6; bottom: -200px; right: -200px; animation-delay: -5s; }
        .premium-bg .blob-3 { width: 500px; height: 500px; background: #1e40af; top: 40%; left: 60%; animation-delay: -12s; }
        @keyframes blob-anim {
            0%, 100% { transform: translate(0, 0) scale(1); }
            33% { transform: translate(30px, -50px) scale(1.1); }
            66% { transform: translate(-20px, 20px) scale(0.9); }
        }
        .premium-bg .noise-overlay {
            position: absolute;
            inset: 0;
            background-image: url("data:image/svg+xml,%3Csvg viewBox='0 0 200 200' xmlns='http://www.w3.org/2000/svg'%3E%3Cfilter id='n'%3E%3CfeTurbulence type='fractalNoise' baseFrequency='0.65' numOctaves='3' stitchTiles='stitch'/%3E%3C/filter%3E%3Crect width='100%25' height='100%25' filter='url(%23n)' opacity='0.03'/%3E%3C/svg%3E");
            mix-blend-mode: overlay;
        }
        .search-container {
            width: 90%;
            max-width: none;
            margin: 20px auto;
            padding: 20px;
            position: relative;
            z-index: 1;
        }
        .glass-card {
            background: rgba(15, 23, 42, 0.6);
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px);
            border: 1px solid rgba(255, 255, 255, 0.08);
            border-radius: 20px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.2);
        }
        .page-header {
            padding: 30px;
            margin-bottom: 30px;
            text-align: center;
        }
        .page-header.glass-card { padding: 30px; }
        .page-header h1 {
            margin-bottom: 10px;
            color: var(--color-white);
            font-family: var(--font-heading);
            font-size: clamp(1.5rem, 3vw, 2rem);
            font-weight: 700;
        }
        .page-header p {
            color: var(--color-muted);
            font-size: 16px;
        }
        .search-section {
            padding: 30px;
            margin-bottom: 30px;
        }
        .search-section.glass-card { padding: 30px; }
        .search-section h2 {
            color: var(--color-white);
            margin-bottom: 20px;
            font-size: 24px;
            font-family: var(--font-heading);
        }
        .skills-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 25px;
        }
        .skill-checkbox {
            display: flex;
            align-items: center;
            padding: 12px;
            background: rgba(255, 255, 255, 0.04);
            border: 1px solid rgba(255, 255, 255, 0.08);
            border-radius: 10px;
            transition: all 0.2s ease;
            cursor: pointer;
        }
        .skill-checkbox:hover {
            background: rgba(96, 165, 250, 0.1);
            border-color: rgba(96, 165, 250, 0.25);
            transform: translateY(-2px);
        }
        .skill-checkbox input[type="checkbox"] {
            margin-right: 10px;
            width: 18px;
            height: 18px;
            cursor: pointer;
            accent-color: var(--royal-blue);
        }
        .skill-checkbox label {
            cursor: pointer;
            flex: 1;
            font-weight: 500;
            color: var(--color-white);
        }
        .skill-checkbox input[type="checkbox"]:checked + label {
            color: var(--royal-blue-light);
            font-weight: 600;
        }
        .search-controls {
            display: flex;
            gap: 15px;
            align-items: center;
            flex-wrap: wrap;
        }
        .btn-search {
            padding: 12px 30px;
            background: var(--color-white);
            color: var(--color-dark);
            border: none;
            border-radius: 50px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 0 20px rgba(255, 255, 255, 0.1);
        }
        .btn-search:hover {
            transform: translateY(-2px);
            box-shadow: 0 0 30px rgba(255, 255, 255, 0.25);
        }
        .btn-select-all, .btn-clear-all {
            padding: 10px 20px;
            background: rgba(255, 255, 255, 0.06);
            border: 1px solid rgba(255, 255, 255, 0.15);
            color: var(--color-white);
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s ease;
        }
        .btn-select-all:hover, .btn-clear-all:hover {
            background: rgba(255, 255, 255, 0.1);
            border-color: rgba(255, 255, 255, 0.25);
        }
        .selected-count {
            padding: 10px 20px;
            background: rgba(59, 130, 246, 0.2);
            color: var(--royal-blue-light);
            border: 1px solid rgba(96, 165, 250, 0.3);
            border-radius: 8px;
            font-weight: 600;
        }
        .results-section {
            padding: 30px;
            margin-bottom: 30px;
        }
        .results-section.glass-card { padding: 30px; }
        .results-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        .results-header h2 {
            color: var(--color-white);
            margin: 0;
            font-family: var(--font-heading);
        }
        .results-count {
            background: var(--royal-blue);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-weight: 600;
        }
        .pdf-card {
            background: rgba(15, 23, 42, 0.5);
            border-radius: 16px;
            padding: 25px;
            margin-bottom: 20px;
            border: 1px solid rgba(255, 255, 255, 0.08);
            transition: all 0.3s ease;
            backdrop-filter: blur(8px);
        }
        .pdf-card:hover {
            border-color: rgba(96, 165, 250, 0.35);
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2), 0 0 20px rgba(59, 130, 246, 0.15);
        }
        .pdf-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 15px;
        }
        .pdf-name {
            font-size: 20px;
            font-weight: 600;
            color: var(--color-white);
            word-break: break-word;
        }
        .match-badge {
            background: var(--gradient-royal-purple);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
            white-space: nowrap;
        }
        .pdf-info {
            display: flex;
            gap: 15px;
            margin-bottom: 15px;
            font-size: 14px;
            color: var(--color-muted);
        }
        .info-item { display: flex; align-items: center; gap: 5px; }
        .skills-section { margin-top: 20px; }
        .skills-section h4 {
            color: var(--color-white);
            margin-bottom: 10px;
            font-size: 16px;
        }
        .skills-list { display: flex; flex-wrap: wrap; gap: 8px; }
        .skill-badge {
            padding: 6px 14px;
            border-radius: 15px;
            font-size: 13px;
            font-weight: 500;
        }
        .skill-matched {
            background: rgba(34, 197, 94, 0.2);
            color: #86efac;
            border: 1px solid rgba(34, 197, 94, 0.4);
        }
        .skill-other {
            background: rgba(96, 165, 250, 0.15);
            color: var(--royal-blue-light);
            border: 1px solid rgba(96, 165, 250, 0.25);
        }
        .download-btn {
            display: inline-block;
            padding: 10px 20px;
            background: var(--royal-blue);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-weight: 600;
            margin-top: 15px;
            transition: all 0.2s ease;
        }
        .download-btn:hover {
            background: var(--royal-blue-dark);
            transform: translateY(-2px);
            box-shadow: 0 4px 20px rgba(59, 130, 246, 0.4);
        }
        .no-results {
            text-align: center;
            padding: 40px;
            color: var(--color-muted);
        }
        .no-results h3 { color: var(--color-white); margin-bottom: 10px; }
        .warning-message {
            background: rgba(30, 58, 138, 0.3);
            border: 1px solid rgba(96, 165, 250, 0.4);
            color: var(--royal-blue-light);
            padding: 20px;
            border-radius: 16px;
            margin-bottom: 20px;
            text-align: center;
        }
        .error-message {
            background: rgba(185, 28, 28, 0.2);
            border: 1px solid rgba(248, 113, 113, 0.5);
            color: #fca5a5;
            padding: 20px;
            border-radius: 16px;
            margin-bottom: 20px;
            text-align: center;
        }
        .back-link {
            display: inline-block;
            margin-top: 20px;
            padding: 12px 24px;
            background: var(--royal-blue);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            transition: all 0.2s ease;
            font-weight: 600;
        }
        .back-link:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 20px rgba(59, 130, 246, 0.4);
        }
        body.search-skills-page .loader-overlay {
            background: rgba(2, 6, 23, 0.95);
        }
        body.search-skills-page .loader-container {
            background: rgba(15, 23, 42, 0.9);
            border: 1px solid rgba(255, 255, 255, 0.1);
            color: var(--color-white);
        }
        body.search-skills-page .loader-spinner {
            border-color: rgba(255, 255, 255, 0.1);
            border-top-color: var(--royal-blue-light);
        }
        body.search-skills-page .loader-text { color: var(--color-white); }
        body.search-skills-page .progress-bar-container { background: rgba(255, 255, 255, 0.1); }
        body.search-skills-page .progress-bar { background: var(--gradient-royal-purple); }
        body.search-skills-page .progress-text { color: var(--color-muted); }
        @media (max-width: 768px) {
            .search-container { padding: 10px; }
            .skills-grid { grid-template-columns: 1fr; }
            .pdf-header { flex-direction: column; gap: 10px; }
            .search-controls { flex-direction: column; align-items: stretch; }
            .btn-search, .btn-select-all, .btn-clear-all { width: 100%; }
        }
    </style>
</head>
<body class="search-skills-page">
    <div class="premium-bg">
        <div class="blob blob-1"></div>
        <div class="blob blob-2"></div>
        <div class="blob blob-3"></div>
        <div class="noise-overlay"></div>
    </div>
    <div class="search-container">
        <div class="page-header">
            <h1>🔍 Search PDFs by Skills</h1>
            <p>Select multiple skills to find matching resumes/documents</p>
            <?php if ($parserAvailable && !empty($pdfFiles)): ?>
                <p style="color: #667eea; font-size: 14px; margin-top: 10px;">
                    📊 Loaded <?php echo count($pdfData); ?> PDFs 
                    (<?php echo $cachedCount; ?> from cache, <?php echo $processedCount; ?> processed)
                    <?php if ($cachedCount > 0): ?>
                        | <a href="?clear_cache=1" style="color: #f44336; text-decoration: none;" 
                           onclick="return confirm('This will rebuild the cache. Continue?');">
                            🔄 Clear Cache
                        </a>
                    <?php endif; ?>
                </p>
            <?php endif; ?>
        </div>
        
        <?php if ($showProgressNotice): ?>
            <div class="warning-message">
                <h3>📊 Processing in Progress</h3>
                <p>Building cache: Processed <?php echo count($pdfData); ?> of <?php echo $totalPDFs; ?> PDFs</p>
                <p><?php echo $remainingPDFs; ?> PDFs remaining. The page will process more on each refresh.</p>
                <?php if (!$searchPerformed): ?>
                    <p style="font-weight: 600; margin-top: 10px;">
                        Auto-refreshing in <span id="countdown">3</span> seconds...
                    </p>
                <?php endif; ?>
                <button onclick="location.reload();" 
                        class="btn-search" style="margin-top: 10px;">
                    🔄 Process Now
                </button>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($skippedFiles)): ?>
            <div class="warning-message">
                <h3>⚠️ Some Files Were Skipped</h3>
                <p>The following PDFs were too large to process (limit: 10MB):</p>
                <ul style="list-style: none; padding: 0; margin-top: 10px;">
                    <?php foreach ($skippedFiles as $skipped): ?>
                        <li style="margin: 5px 0;">
                            <strong><?php echo htmlspecialchars($skipped['name']); ?></strong> 
                            - <?php echo $skipped['size']; ?>
                        </li>
                    <?php endforeach; ?>
                </ul>
                <p style="margin-top: 15px; font-size: 13px;">
                    💡 Tip: Try compressing these PDFs or contact your administrator to increase the memory limit.
                </p>
            </div>
        <?php endif; ?>
        
        <?php if (!$parserAvailable): ?>
            <div class="error-message">
                <h3>⚠️ PDF Parser Not Installed</h3>
                <p>To use this feature, you need to install the PDF parser library.</p>
                <p>Run: <code>composer install</code></p>
            </div>
        <?php elseif (empty($pdfFiles)): ?>
            <div class="warning-message">
                <h3>📁 No PDF Files Found</h3>
                <p>Please add PDF files to the <strong>pdfs</strong> folder.</p>
            </div>
        <?php elseif (empty($allSkills)): ?>
            <div class="warning-message">
                <h3>⚠️ No Skills Found</h3>
                <p>No skills were detected in the PDF files.</p>
            </div>
        <?php else: ?>
            <div class="search-section glass-card">
                <h2>🎯 Select Skills to Search</h2>
                
                <form method="POST" id="searchForm">
                    <div class="skills-grid" id="skillsGrid">
                        <?php foreach ($allSkills as $skill): ?>
                            <div class="skill-checkbox">
                                <input 
                                    type="checkbox" 
                                    name="skills[]" 
                                    value="<?php echo htmlspecialchars($skill); ?>" 
                                    id="skill_<?php echo htmlspecialchars($skill); ?>"
                                    <?php echo in_array($skill, $selectedSkills) ? 'checked' : ''; ?>
                                    onchange="updateCount()"
                                >
                                <label for="skill_<?php echo htmlspecialchars($skill); ?>">
                                    <?php echo htmlspecialchars($skill); ?>
                                </label>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <div class="search-controls">
                        <button type="submit" name="search" class="btn-search">
                            🔍 Search PDFs
                        </button>
                        <button type="button" class="btn-select-all" onclick="selectAll()">
                            ✓ Select All
                        </button>
                        <button type="button" class="btn-clear-all" onclick="clearAll()">
                            ✗ Clear All
                        </button>
                        <div class="selected-count">
                            Selected: <span id="selectedCount">0</span>
                        </div>
                    </div>
                </form>
            </div>
            
            <?php if ($searchPerformed): ?>
                <div class="results-section glass-card">
                    <div class="results-header">
                        <h2>📋 Search Results</h2>
                        <div class="results-count"><?php echo count($searchResults); ?> PDF(s) Found</div>
                    </div>
                    
                    <?php if (empty($selectedSkills)): ?>
                        <div class="no-results">
                            <h3>Please select at least one skill</h3>
                            <p>Check the boxes above and click "Search PDFs"</p>
                        </div>
                    <?php elseif (empty($searchResults)): ?>
                        <div class="no-results">
                            <h3>No matching PDFs found</h3>
                            <p>Try selecting different skills or fewer skills at once</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($searchResults as $result): ?>
                            <div class="pdf-card">
                                <div class="pdf-header">
                                    <div class="pdf-name">
                                        📄 <?php echo htmlspecialchars($result['pdf']['name']); ?>
                                    </div>
                                    <div class="match-badge">
                                        <?php echo $result['matchCount']; ?>/<?php echo count($selectedSkills); ?> Skills Matched
                                        (<?php echo round($result['matchPercentage']); ?>%)
                                    </div>
                                </div>
                                
                                <div class="pdf-info">
                                    <div class="info-item">
                                        <strong>Size:</strong> <?php echo $result['pdf']['size']; ?>
                                    </div>
                                    <div class="info-item">
                                        <strong>Total Skills:</strong> <?php echo $result['pdf']['skillCount']; ?>
                                    </div>
                                </div>
                                
                                <div class="skills-section">
                                    <h4>✓ Matched Skills (<?php echo count($result['matchingSkills']); ?>):</h4>
                                    <div class="skills-list">
                                        <?php foreach ($result['matchingSkills'] as $skill): ?>
                                            <span class="skill-badge skill-matched">
                                                ✓ <?php echo htmlspecialchars($skill); ?>
                                            </span>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                                
                                <?php 
                                $otherSkills = array_diff($result['pdf']['skills'], $result['matchingSkills']);
                                if (!empty($otherSkills)): 
                                ?>
                                    <div class="skills-section" style="margin-top: 15px;">
                                        <h4>Other Skills (<?php echo count($otherSkills); ?>):</h4>
                                        <div class="skills-list">
                                            <?php foreach ($otherSkills as $skill): ?>
                                                <span class="skill-badge skill-other">
                                                    <?php echo htmlspecialchars($skill); ?>
                                                </span>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                <?php endif; ?>
                                
                                <a href="<?php echo htmlspecialchars($result['pdf']['path']); ?>" 
                                   class="download-btn" 
                                   download
                                   target="_blank">
                                    📥 Download PDF
                                </a>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>
        
        <div style="text-align: center;">
            <a href="view_skills.php" class="back-link" style="margin-left: 10px;">📊 View All Skills</a>
        </div>
    </div>
    
    <!-- Loader Overlay -->
    <div class="loader-overlay active" id="loaderOverlay">
        <div class="loader-container">
            <div class="loader-spinner"></div>
            <div class="loader-text" id="loaderText">Loading Skills from PDFs...</div>
            <div class="progress-bar-container">
                <div class="progress-bar"></div>
            </div>
            <div class="progress-text">This may take a few moments, please wait...</div>
        </div>
    </div>
    
    <script>
        // Auto-reload if there are PDFs remaining to process
        <?php if ($showProgressNotice && !$searchPerformed): ?>
        var countdown = 3;
        var countdownElement = document.getElementById('countdown');
        
        var countdownInterval = setInterval(function() {
            countdown--;
            if (countdownElement) {
                countdownElement.textContent = countdown;
            }
            if (countdown <= 0) {
                clearInterval(countdownInterval);
                location.reload();
            }
        }, 1000);
        <?php endif; ?>
        
        // Show loader immediately when page starts loading
        window.addEventListener('load', function() {
            // Hide loader once page is fully loaded
            const loader = document.getElementById('loaderOverlay');
            if (loader) {
                // Small delay to ensure smooth transition
                setTimeout(function() {
                    loader.classList.remove('active');
                }, 300);
            }
        });
        
        // Update selected count on page load
        document.addEventListener('DOMContentLoaded', function() {
            updateCount();
        });
        
        function updateCount() {
            const checkboxes = document.querySelectorAll('input[name="skills[]"]:checked');
            const count = checkboxes.length;
            document.getElementById('selectedCount').textContent = count;
        }
        
        function selectAll() {
            const checkboxes = document.querySelectorAll('input[name="skills[]"]');
            checkboxes.forEach(checkbox => {
                checkbox.checked = true;
            });
            updateCount();
        }
        
        function clearAll() {
            const checkboxes = document.querySelectorAll('input[name="skills[]"]');
            checkboxes.forEach(checkbox => {
                checkbox.checked = false;
            });
            updateCount();
        }
        
        // Show loader on form submission
        const searchForm = document.getElementById('searchForm');
        if (searchForm) {
            searchForm.addEventListener('submit', function(e) {
                const selectedCount = document.querySelectorAll('input[name="skills[]"]:checked').length;
                
                if (selectedCount === 0) {
                    e.preventDefault();
                    alert('Please select at least one skill to search.');
                    return false;
                }
                
                const loader = document.getElementById('loaderOverlay');
                const loaderText = document.getElementById('loaderText');
                const progressText = document.querySelector('.progress-text');
                
                loaderText.textContent = 'Searching PDFs for Matching Skills...';
                progressText.textContent = 'Analyzing ' + selectedCount + ' skill(s) across all PDFs...';
                loader.classList.add('active');
            });
        }
        
        // Show loader on navigation links
        const links = document.querySelectorAll('a[href$=".php"]');
        links.forEach(link => {
            link.addEventListener('click', function(e) {
                const loader = document.getElementById('loaderOverlay');
                const loaderText = document.getElementById('loaderText');
                const progressText = document.querySelector('.progress-text');
                
                loaderText.textContent = 'Loading page...';
                progressText.textContent = 'Please wait...';
                loader.classList.add('active');
            });
        });
    </script>
</body>
</html>

