// Signature Canvas Functionality
const canvas = document.getElementById('signatureCanvas');
const ctx = canvas.getContext('2d');
const clearBtn = document.getElementById('clearSignature');
const form = document.getElementById('registrationForm');
const signatureInput = document.getElementById('signatureData');

// Set canvas size
function resizeCanvas() {
    const container = canvas.parentElement;
    canvas.width = container.offsetWidth;
    canvas.height = 200;
    
    // Set drawing styles
    ctx.strokeStyle = '#000000';
    ctx.lineWidth = 2;
    ctx.lineCap = 'round';
    ctx.lineJoin = 'round';
}

// Initialize canvas
resizeCanvas();
window.addEventListener('resize', resizeCanvas);

// Drawing state
let isDrawing = false;
let lastX = 0;
let lastY = 0;
let hasDrawn = false;

// Get coordinates relative to canvas
function getCoordinates(e) {
    const rect = canvas.getBoundingClientRect();
    const scaleX = canvas.width / rect.width;
    const scaleY = canvas.height / rect.height;
    
    if (e.type.includes('touch')) {
        return {
            x: (e.touches[0].clientX - rect.left) * scaleX,
            y: (e.touches[0].clientY - rect.top) * scaleY
        };
    } else {
        return {
            x: (e.clientX - rect.left) * scaleX,
            y: (e.clientY - rect.top) * scaleY
        };
    }
}

// Start drawing
function startDrawing(e) {
    isDrawing = true;
    const coords = getCoordinates(e);
    lastX = coords.x;
    lastY = coords.y;
    e.preventDefault();
}

// Draw on canvas
function draw(e) {
    if (!isDrawing) return;
    
    const coords = getCoordinates(e);
    
    ctx.beginPath();
    ctx.moveTo(lastX, lastY);
    ctx.lineTo(coords.x, coords.y);
    ctx.stroke();
    
    lastX = coords.x;
    lastY = coords.y;
    hasDrawn = true;
    
    e.preventDefault();
}

// Stop drawing
function stopDrawing() {
    if (isDrawing) {
        isDrawing = false;
        // Save signature data when user stops drawing
        saveSignature();
    }
}

// Save signature as base64
function saveSignature() {
    if (hasDrawn) {
        const signatureData = canvas.toDataURL('image/png');
        signatureInput.value = signatureData;
    }
}

// Clear signature
function clearSignature() {
    ctx.clearRect(0, 0, canvas.width, canvas.height);
    signatureInput.value = '';
    hasDrawn = false;
}

// Mouse events
canvas.addEventListener('mousedown', startDrawing);
canvas.addEventListener('mousemove', draw);
canvas.addEventListener('mouseup', stopDrawing);
canvas.addEventListener('mouseout', stopDrawing);

// Touch events for mobile devices
canvas.addEventListener('touchstart', startDrawing);
canvas.addEventListener('touchmove', draw);
canvas.addEventListener('touchend', stopDrawing);

// Clear button
clearBtn.addEventListener('click', clearSignature);

// Form submission validation
form.addEventListener('submit', function(e) {
    if (!hasDrawn || !signatureInput.value) {
        e.preventDefault();
        alert('Please provide your signature before submitting the form.');
        return false;
    }
    
    // Show loader
    const loader = document.getElementById('loaderOverlay');
    const loaderText = document.getElementById('loaderText');
    if (loader && loaderText) {
        loaderText.textContent = 'Submitting form...';
        loader.classList.add('active');
    }
    
    return true;
});

